from __future__ import print_function

import time
import datetime
import logging
from sandbox import sdk2
import sandbox.common.types.task as ctt
import sandbox.common.types.notification as ctn

from .. import common as eviction
from ..QypEvacuate import QypEvacuate

import pytz

bad_events = [
    ctt.Status.NOT_RELEASED,
    ctt.Status.FAILURE,
    ctt.Status.DELETED,
    ctt.Status.Group.BREAK  # exception, timeout, stopped, expired, no_res
]
# DC_LIST = ('sas', 'myt', 'iva', 'vla', 'man')
DC_LIST = ('sas', )


class QypEvacuatePlanner(sdk2.Task):
    class Parameters(sdk2.Task.Parameters):
        dryrun = sdk2.parameters.Bool(
            'Dry run (dont run evacuate tasks)', required=True, default=False
        )

        notifications = [
            # Bad events
            sdk2.Notification(
                bad_events,
                ['mocksoul', 'golomolzin', 'frolstas'],
                ctn.Transport.EMAIL
            ),

            # Only very bad events
            sdk2.Notification(
                bad_events,
                ['mocksoul', 'golomolzin', 'frolstas'],
                ctn.Transport.TELEGRAM
            )
        ]

    def _human_secs(self, s):
        ret = []
        days = s // 86400
        hours = (s - (days * 86400)) // 3600
        minutes = (s - (days * 86400) - (hours * 3600)) // 60
        seconds = (s - (days * 86400) - (hours * 3600) - (minutes * 60))

        if days > 0:
            ret.append('%dd' % (days, ))

        if hours or ret:
            ret.append(('%02dh' % (hours, )) if ret else '%dh' % (hours, ))

        if days == 0:
            # If we have days -- do not report minutes/secs at all
            if minutes or ret:
                ret.append(('%02dm' % (minutes, )) if ret else '%dm' % (minutes, ))

            if hours == 0:
                if ret:
                    ret.append('%02ds' % (seconds, ))
                else:
                    ret.append('%ds' % (seconds, ))

        return ''.join(ret)

    def on_execute(self):
        yp_oauth = sdk2.Vault.data('QYP_YP_TOKEN')
        dryrun = self.Parameters.dryrun

        for dc in DC_LIST:
            pods_for_eviction = eviction.get_pods_for_eviction(dc, yp_oauth_token=yp_oauth, production=True)

            ignore_pod_idxs = []

            for idx, pod_info in enumerate(pods_for_eviction):
                logging.info('[%s][%s] Analyzing eviction request: %r', dc, pod_info['id'], pod_info['eviction'])

                if (
                    pod_info['eviction']['reason'] == 'scheduler' or
                    (
                        pod_info['eviction']['reason'] == 'client' and                # QEMUKVM-1116
                        'cluster defragmentation' in pod_info['eviction']['message']  # QEMUKVM-1116
                    )
                ):
                    scheduled_secs_ago = int(time.time() - (pod_info['scheduling_last_updated'] / 10 ** 6))
                    scheduled_days_ago = int(scheduled_secs_ago / 86400)

                    if scheduled_days_ago < 30:
                        logging.info(
                            '  pod scheduled %d days (%d secs) ago, <30, ignoring eviction request',
                            scheduled_days_ago, scheduled_secs_ago
                        )
                        ignore_pod_idxs.append(idx)
                    else:
                        logging.info('  pod scheduled %d days ago, eviction allowed', scheduled_days_ago)

            for idx in reversed(ignore_pod_idxs):
                del pods_for_eviction[idx]

            pod_ids_by_node = {}
            for pod_info in pods_for_eviction:
                pod_ids_by_node.setdefault(pod_info['node_id'], {})[pod_info['id']] = (
                    None, None, None, None, None  # task, new, enqueued, status, last_update
                )

            info_text = ['[%s] Need to evacuate:' % (dc, )]

            for pod_info in pods_for_eviction:
                logging.info('Need to evacuate pod "%s" (%s)', pod_info['id'], pod_info['node_id'])

                evacuate_task_id = eviction.get_pod_eviction_label(dc, yp_oauth, pod_info['id'], '/sb_task')

                if evacuate_task_id:
                    logging.debug('Found evacuation task id %r', evacuate_task_id)
                    evacuate_task = QypEvacuate.find(id=evacuate_task_id).limit(1).first()

                    if not evacuate_task:
                        logging.info('Unable to find old evacuate task (got %r)', evacuate_task)
                    else:
                        if evacuate_task.status is ctt.Status.SUCCESS:
                            logging.info(
                                'Current evacuate task in finished state, but we stil want '
                                'to schedule evacuation, do it now'
                            )

                        elif evacuate_task.status in (
                            ctt.Status.FAILURE, ctt.Status.EXCEPTION, ctt.Status.DELETED,
                            ctt.Status.TIMEOUT, ctt.Status.STOPPED
                        ):
                            # FAILURE: known error, retry in 1 day (avoid to much mail spamming)
                            # EXCEPTION: unknown error, retry in 1.5 days (4 tries total in 6 days)
                            # TIMEOUT: unable to finish, retry immidiately if pod still has eviction request
                            # STOPPED: stopped by hand, do not retry in this eviction request window (10d)
                            # DELETED: deleted by hand, do not retry in this eviction request window (10d)
                            min_retries = {
                                ctt.Status.FAILURE: datetime.timedelta(days=1),
                                ctt.Status.EXCEPTION: datetime.timedelta(days=1, hours=12),
                                ctt.Status.TIMEOUT: datetime.timedelta(seconds=1),
                                ctt.Status.STOPPED: datetime.timedelta(days=10),
                                ctt.Status.DELETED: datetime.timedelta(days=10),
                            }

                            min_retry = min_retries[evacuate_task.status]

                            if evacuate_task.updated.tzinfo:
                                now = datetime.datetime.utcnow().replace(tzinfo=pytz.UTC)
                            else:
                                now = datetime.datetime.now()

                            if now - evacuate_task.updated > min_retry:
                                if evacuate_task.status not in ctt.Status.Group.FINISH:
                                    logging.info(
                                        'Current evacuate task has status %r, and min retry time was passed, '
                                        'will resume old task',
                                        evacuate_task.status
                                    )

                                    old_evacuate_task_updated = evacuate_task.updated
                                    evacuate_task.enqueue()

                                    pod_ids_by_node[pod_info['node_id']][pod_info['id']] = (
                                        evacuate_task_id, False, True, evacuate_task.status, old_evacuate_task_updated
                                    )
                                    continue
                                else:
                                    logging.info(
                                        'Current evacuate task has status %r, and min retry time was passed, '
                                        'but we cant resume old task, creating new instead',
                                        evacuate_task.status
                                    )
                            else:
                                logging.info(
                                    'Current evacuate task has status %r, but min retry time was not passed yet, '
                                    'will not create new task'
                                )
                                pod_ids_by_node[pod_info['node_id']][pod_info['id']] = (
                                    evacuate_task_id, False, False, evacuate_task.status, evacuate_task.updated
                                )
                                continue

                        else:
                            # All other intermediate statuses, we need to wait more
                            logging.info('Current evacuate task not in finished/failed state, wait it')
                            pod_ids_by_node[pod_info['node_id']][pod_info['id']] = (
                                evacuate_task_id, False, False, evacuate_task.status, evacuate_task.updated
                            )
                            continue

                if not dryrun:
                    eviction.init_qyp_eviction_label(dc, yp_oauth, pod_info['id'])
                    evacuate_task = QypEvacuate(
                        None,
                        description='QYP evacuation task of pod "%s" from node "%s"' % (
                            pod_info['id'], pod_info['node_id']
                        ),
                        owner=self.owner,
                        pod_id=pod_info['id'],
                        vmproxy_dc=dc
                    )

                    evacuate_task.save()
                    eviction.update_pod_eviction_label(dc, yp_oauth, pod_info['id'], evacuate_task.id, '/sb_task')
                    evacuate_task.enqueue()

                    pod_ids_by_node[pod_info['node_id']][pod_info['id']] = (
                        evacuate_task.id, True, True, 'new', datetime.datetime.utcnow().replace(tzinfo=pytz.UTC)
                    )
                else:
                    pod_ids_by_node[pod_info['node_id']][pod_info['id']] = (
                        None, None, None, None, None
                    )

            if pod_ids_by_node:
                for node, pods in pod_ids_by_node.items():
                    info_text.append('  %s:' % (node, ))

                    for pod_id, (task_id, created, enqueued, task_status, task_updated) in pods.items():
                        if task_id:
                            if created:
                                info_text.append(
                                    '    %s [<a href="/task/%d">%d</a> (%s)]' % (
                                        pod_id, task_id, task_id, task_status
                                    )
                                )
                            else:
                                if task_updated.tzinfo:
                                    now = datetime.datetime.utcnow().replace(tzinfo=pytz.UTC)
                                else:
                                    now = datetime.datetime.now()

                                task_updated_secs = (now - task_updated).total_seconds()

                                info_text.append(
                                    '    %s [<a href="/task/%d">%d</a> '
                                    '(%s, old, %s, updated %s ago)]' % (
                                        pod_id, task_id, task_id, task_status,
                                        'enqueued again' if enqueued else 'waiting',
                                        self._human_secs(task_updated_secs)
                                    )
                                )
                        else:
                            info_text.append('    %s' % (pod_id, ))

                    info_text.append('')

                self.set_info('<br/>'.join(info_text), do_escape=False)
            else:
                self.set_info('[%s] No pods needed evacuation found' % (dc, ))

        if dryrun:
            logging.info('DRYRUN MODE: NO TASKS WERE MADE')
