# -*- coding: utf-8 -*-
from __future__ import absolute_import, division, print_function, unicode_literals

import logging
import os
import shutil

import sandbox.common.types.task as ctt
import sandbox.sdk2.helpers
from sandbox import common
from sandbox import sdk2

from sandbox.projects.BuildDockerImageFromGit import DockerContentResourse
from sandbox.projects.BuildDockerImageV6 import BuildDockerImageV6
from sandbox.projects.rasp.qloud.RestartEnvironment import RaspQloudRestartEnvironment
from sandbox.projects.rasp.utils.changelog import generate_changelog
from sandbox.projects.rasp.utils.email_notifications import EmailNotificationMixin, use_email_notification_params
from sandbox.projects.rasp.utils.retry_task import retry_task_on
from sandbox.sdk2.helpers import subprocess as sp


RELEASE_TYPE_CHOICES = [
    ('Major', 'major'),
    ('Minor', 'minor'),
    ('Fix', 'fix'),
    ('Experimental', 'experimental'),
]


GIT_TIMEOUT = 1200  # morda/markup клонируется долго
HOOKFILE = 'sandbox_predocker.sh'
HOOK_TIMEOUT = 600


class RaspBuildAndRelease(sdk2.Task, EmailNotificationMixin):
    """
    Fork BuildDockerImageFromGit для Расписаний
    """

    checkout_path = 'src'

    class Parameters(sdk2.Task.Parameters):
        kill_timeout = 3600

        with sdk2.parameters.Group('Version parameters') as versions_block:
            release_type = sdk2.parameters.String('Release Type', required=True, choices=RELEASE_TYPE_CHOICES,
                                                  default='minor')
            experiment_tag = sdk2.parameters.String('Experiment Tag', required=False, default='')
            version_tag_mark = sdk2.parameters.String('Version Tag Mark', required=False, default='')
            last_version_environment_name = sdk2.parameters.String(
                'Name of the environment, from which latest version tag will be taken',
                required=False, default='production')

        with sdk2.parameters.Group('Git parameters') as git_block:
            repository = sdk2.parameters.String('Repository', required=True)
            branch = sdk2.parameters.String('Branch', default='master')
            path_to_dockerfile = sdk2.parameters.String('Path to dockerfile inside the repository', default='.')
            dockerfile_name = sdk2.parameters.String(
                "Dockerfile name (defaults to 'Dockerfile')", default="Dockerfile"
            )
            ssh_vault_name = sdk2.parameters.String('Vault item with ssh key for git access')
            ssh_vault_owner = sdk2.parameters.String('Vault item owner')

        with sdk2.parameters.Group('Registry parameters') as docker_block:
            registry_url = sdk2.parameters.String('Registry Url to publish image with, tag autogenerated'
                                                  ' (registry.yandex.net/<this url>:<tag>)')
            oauth_vault_name = sdk2.parameters.String(
                'Vault item with oauth token for '
                'registry.yandex.net (vault item name)'
            )
            oauth_vault_owner = sdk2.parameters.String('Vault item owner')
            registry_login = sdk2.parameters.String('Yandex login to use with docker login')

        with sdk2.parameters.Group('Qloud Restart') as qloud_block:
            qloud_component_paths = sdk2.parameters.String('Qloud components to restart, comma separated, '
                                                           'use asteriks for wildcards, for example: '
                                                           'rasp.blablacar.testing.backend,rasp.blablacar.static.*')

        _email_notification_params = use_email_notification_params()

    def git_clone(self):
        if os.path.exists(self.checkout_path):
            shutil.rmtree(self.checkout_path)
        os.makedirs(self.checkout_path)

        with sandbox.sdk2.helpers.ProcessLog(self, logger='git-clone') as pl:
            check_call_kwargs = {'stdout': pl.stdout, 'stderr': pl.stderr, 'cwd': self.checkout_path,
                                 'timeout': GIT_TIMEOUT}
            sp.check_call(['git', 'init'], **check_call_kwargs)
            sp.check_call(['git', 'remote', 'add', 'origin', self.Parameters.repository], **check_call_kwargs)

            remote_commit, full_ref = sp.Popen(
                ['git', 'ls-remote', '--heads', '--tags', 'origin', self.Parameters.branch],
                stdout=sp.PIPE, stderr=pl.stderr, cwd=self.checkout_path,
            ).communicate(timeout=GIT_TIMEOUT)[0].strip().split(None, 1)
            sp.check_call(['git', 'fetch', 'origin', full_ref], **check_call_kwargs)
            sp.check_call(['git', 'update-ref', full_ref, remote_commit], **check_call_kwargs)
            sp.check_call(['git', 'checkout', '-f', self.Parameters.branch], **check_call_kwargs)

            # !Обязательно нужно фетчить теги, версия гита страрая(~1.7) и перепушивает теги без форса
            # Кроме того, это нужно для генерации changelog
            sp.check_call(['git', 'fetch', '--tags', 'origin'], **check_call_kwargs)
            sp.check_call(['git', 'submodule', 'update', '--init'], **check_call_kwargs)

    def gen_tag(self):
        new_tag_params = (
            ['--experiment-tag={}'.format(self.Parameters.experiment_tag)]
            if self.Parameters.experiment_tag
            else []
        )
        with sandbox.sdk2.helpers.ProcessLog(self, logger='gen-tag') as pl:
            check_call_kwargs = {'stdout': pl.stdout, 'stderr': pl.stderr, 'cwd': self.checkout_path,
                                 'timeout': 120}
            sp.check_call(['python', './deploy_tools/versions.py', 'new-tag',
                           '--remote={}'.format(self.Parameters.repository),
                           '--release-type={}'.format(self.Parameters.release_type),
                           ] + new_tag_params + self.version_tag_mark_params, **check_call_kwargs)
            new_tag = sp.check_output(
                ['python', './deploy_tools/versions.py', 'get-version'] + self.version_tag_mark_params,
                cwd=self.checkout_path,
                timeout=120,
                stderr=pl.stderr,
            ).strip()
            if not new_tag:
                logging.error('Fail getting new tag')
                raise Exception('Fail getting new tag')
            logging.info('New version is %s', new_tag)
            registry_url = (
                self.Parameters.registry_url
                if self.Parameters.registry_url.startswith('registry.yandex.net/')
                else 'registry.yandex.net/' + self.Parameters.registry_url
            )
            self.Context.registry_tag = registry_url + ':' + new_tag

            logging.info('Build changelog')

    def run_predocker_hook(self, dir_path):
        hook_file = os.path.join(dir_path, HOOKFILE)
        if os.path.exists(hook_file):
            with sandbox.sdk2.helpers.ProcessLog(self, logger='predocker-hook') as pl:
                sp.check_call(['/usr/bin/env', 'bash', './' + HOOKFILE],
                              stdout=pl.stdout, stderr=pl.stderr, timeout=HOOK_TIMEOUT, cwd=dir_path)

    def prepare_resource(self):
        dir_path = os.path.join(self.checkout_path, self.Parameters.path_to_dockerfile)
        self.run_predocker_hook(dir_path)

        dockerfile_path = os.path.join(dir_path, self.Parameters.dockerfile_name)
        if not os.path.exists(dockerfile_path):
            raise common.errors.TaskError("Dockerfile {} not found inside the repository".format(
                dockerfile_path)
            )
        # make sure dockerfile is named 'Dockerfile' so that the child task could fetch it
        shutil.move(dockerfile_path, os.path.join(dir_path, 'Dockerfile'))
        resource = DockerContentResourse(
            self,
            "Dockerfile and contents from {repo}".format(
                repo=self.Parameters.repository,
            ),
            dir_path
        )
        sdk2.ResourceData(resource).ready()
        return resource

    def build_image(self, resource):
        task_class = sdk2.Task['BUILD_DOCKER_IMAGE_V6']
        kwargs = {
            BuildDockerImageV6.PackagedResource.name: resource.id,
            BuildDockerImageV6.RegistryTags.name: [self.Context.registry_tag],
            BuildDockerImageV6.RegistryLogin.name: self.Parameters.registry_login,
            BuildDockerImageV6.VaultItemName.name: self.Parameters.oauth_vault_name,
            BuildDockerImageV6.VaultItemOwner.name: self.Parameters.oauth_vault_owner
        }
        sub_task = task_class(
            task_class.current,
            description="Building and publishing Docker image from Git source (by {})".format(self.id),
            owner=self.Parameters.owner,
            priority=self.Parameters.priority,
            **kwargs
        ).enqueue()
        self.Context.sub_task_id = sub_task.id
        raise sdk2.WaitTask([sub_task], ctt.Status.Group.FINISH | ctt.Status.Group.BREAK, wait_all=True)

    def check_subtasks(self):
        if not all(task.status == ctt.Status.SUCCESS for task in self.find()):
            raise common.errors.TaskFailure('Some of child tasks failed')

    @property
    def version_tag_mark_params(self):
        if not hasattr(self, '_version_tag_mark_params'):
            self._version_tag_mark_params = (
                ['--version-tag-mark={}'.format(self.Parameters.version_tag_mark)]
                if self.Parameters.version_tag_mark
                else []
            )
        return self._version_tag_mark_params

    @retry_task_on((sp.CalledProcessError, common.errors.TaskFailure))
    def on_execute(self):
        components = [env.strip() for env in self.Parameters.qloud_component_paths.split(',') if env]
        with self.memoize_stage.first_step:
            with sdk2.ssh.Key(self, self.Parameters.ssh_vault_owner, self.Parameters.ssh_vault_name):
                self.git_clone()
                self.gen_tag()
                self.Context.changelog = generate_changelog(
                    self.Parameters.oauth_vault_owner,
                    self.Parameters.oauth_vault_name,
                    components,
                    self.Parameters.last_version_environment_name,
                    self.version_tag_mark_params,
                    self.checkout_path
                )
            logging.info('Building docker image, publish to %s', self.Context.registry_tag)
            logging.info(self.Context.changelog)
            resource = self.prepare_resource()
            self.build_image(resource)

        with self.memoize_stage.check_build_task:
            self.check_subtasks()

        with self.memoize_stage.second_step:
            if not components:
                logging.info('There are no components to restart. Done.')
            else:
                logging.info('Restart components %r', components)
                raise sdk2.WaitTask(
                    [
                        RaspQloudRestartEnvironment(
                            self,
                            description='Updating qloud component. Triggered by TeamCity.',
                            qloud_component_path=component,
                            comment=self.Context.changelog,
                            registry_tag=self.Context.registry_tag,
                            token_name=self.Parameters.oauth_vault_name,
                            token_owner=self.Parameters.ssh_vault_owner,

                            priority=self.Parameters.priority,

                            enable_email_notifications=False
                        ).enqueue() for component in components
                    ],
                    ctt.Status.Group.FINISH + ctt.Status.Group.BREAK,
                    wait_all=True,
                )

        if components:
            with self.memoize_stage.check_restart_task:
                self.check_subtasks()

    def on_save(self):
        super(RaspBuildAndRelease, self).on_save()
        self.add_email_notifications()
