# coding: utf8
from __future__ import absolute_import, division, print_function, unicode_literals

import sandbox.common.types.task as ctt
from sandbox import common
from sandbox import sdk2
from sandbox.projects.rasp.qloud.RestartEnvironment import RaspQloudRestartEnvironment
from sandbox.projects.rasp.utils.email_notifications import EmailNotificationMixin, use_email_notification_params
from sandbox.sandboxsdk.svn import Arcadia


class RaspBuildDockerImageFromArcadiaAndDeployItToDocker(sdk2.Task, EmailNotificationMixin):
    """
    Собираем релиз из Аркадии и выкатываем в Qloud
    """

    class Parameters(sdk2.Task.Parameters):
        kill_timeout = 3600

        with sdk2.parameters.Group('Arcadia') as versions_block:
            path_to_ya_package_config = sdk2.parameters.String('Path to YA_PACKAGE config', required=True)

        with sdk2.parameters.Group('Docker Registry Config') as docker_block:
            docker_service_name = sdk2.parameters.String('Service Name (like rasp)')
            docker_image_name = sdk2.parameters.String('Image Name (like experimental-route-search-api)')
            docker_tag = sdk2.parameters.String('Tag Template (like 0.0.{revision})')

            docker_vault_owner = sdk2.parameters.String('Vault item owner')
            docker_vault_name = sdk2.parameters.String(
                'Vault item with oauth token for '
                'registry.yandex.net (vault item name)'
            )
            docker_registry_login = sdk2.parameters.String('Yandex login to use with docker login')

        with sdk2.parameters.Group('Qloud Config') as qloud_block:
            qloud_service_name = sdk2.parameters.String('Qloud Service (like rasp)')
            qloud_project_name = sdk2.parameters.String('Qloud Project (like experimental-route-search-api)')
            qloud_environment_name = sdk2.parameters.String('Qloud Environment (like unstable)')
            qloud_component_name = sdk2.parameters.String('Qloud Component (like main)')
            qloud_vault_name = sdk2.parameters.String('Qloud vault name')
            qloud_vault_owner = sdk2.parameters.String('Qloud vault owner')

        _email_notification_params = use_email_notification_params()

    def on_execute(self):
        with self.memoize_stage.build_and_publish_docker_image:
            revision = Arcadia.info('arcadia:/arc/trunk/arcadia')['commit_revision']
            docker_tag = "registry.yandex.net/{service}/{image}:{tag}".format(
                service=self.Parameters.docker_service_name,
                image=self.Parameters.docker_image_name,
                tag=self.Parameters.docker_tag.replace('{revision}', revision)
            )
            self.Context.docker_tag = docker_tag
            build_docker_image_task = sdk2.Task['BUILD_DOCKER_IMAGE_V6'](
                self,
                owner=self.Parameters.owner,
                priority=self.Parameters.priority,
                description="Building and publishing Docker image from Arcadia source (by {})".format(self.id),

                #  Arcadia
                arcadia_url="arcadia:/arc/trunk/arcadia@{revision}",
                docker_package_json=self.Parameters.path_to_ya_package_config,
                use_arcadia_api_fuse=True,

                # Docker Registry
                registry_tags=[docker_tag],
                vault_item_owner=self.Parameters.docker_vault_owner,
                vault_item_name=self.Parameters.docker_vault_name,
                registry_login=self.Parameters.docker_registry_login,
            )

            build_docker_image_task.enqueue()
            raise sdk2.WaitTask(
                [build_docker_image_task],
                ctt.Status.Group.FINISH | ctt.Status.Group.BREAK,
                wait_all=True,
            )

        with self.memoize_stage.check_build_and_publish_docker_image:
            self.__check_subtasks('BuildDockerImageV6 has failed')

        with self.memoize_stage.deploy_to_qloud:
            qloud_component_path = '{service}.{project}.{environment}.{component}'.format(
                service=self.Parameters.qloud_service_name,
                project=self.Parameters.qloud_project_name,
                environment=self.Parameters.qloud_environment_name,
                component=self.Parameters.qloud_component_name
            )
            rasp_qloud_restart_environment_task = RaspQloudRestartEnvironment(
                self,
                owner=self.Parameters.owner,
                priority=self.Parameters.priority,
                description="Deploy {} to {}".format(self.Context.docker_tag, qloud_component_path),

                qloud_component_path=qloud_component_path,
                comment='New version',
                registry_tag=self.Context.docker_tag,
                token_name=self.Parameters.qloud_vault_name,
                token_owner=self.Parameters.qloud_vault_owner,

                enable_email_notifications=False
            )

            rasp_qloud_restart_environment_task.enqueue()
            raise sdk2.WaitTask(
                [rasp_qloud_restart_environment_task],
                ctt.Status.Group.FINISH | ctt.Status.Group.BREAK,
                wait_all=True,
            )

        with self.memoize_stage.check_deploy_to_qloud:
            self.__check_subtasks('RaspQloudRestartEnvironment has failed')

    def __check_subtasks(self, message):
        if not all(task.status == ctt.Status.SUCCESS for task in self.find()):
            raise common.errors.TaskFailure(message)

    def on_save(self):
        super(RaspBuildDockerImageFromArcadiaAndDeployItToDocker, self).on_save()
        self.add_email_notifications()
