# -*- coding: utf-8 -*-
from __future__ import absolute_import, division, print_function, unicode_literals

import logging
from datetime import timedelta

from sandbox import common, sdk2
from sandbox.projects.common import binary_task
from sandbox.projects.rasp.utils.email_notifications import EmailNotificationMixin, use_email_notification_params
from sandbox.projects.Travel.resources.dicts import is_rasp_dict

log = logging.getLogger()


class RaspCheckRaspDataResources(
    binary_task.LastBinaryTaskRelease,
    sdk2.Task,
    EmailNotificationMixin
):
    class Parameters(sdk2.Task.Parameters):
        kill_timeout = timedelta(hours=1).total_seconds()

        with sdk2.parameters.Group('General') as general:
            resources = sdk2.parameters.Resource('Resources to check', multiple=True)

        _binary_task_params = binary_task.binary_release_parameters(stable=True)
        _email_notification_params = use_email_notification_params()

    def on_save(self):
        binary_task.LastBinaryTaskRelease.on_save(self)
        self.add_email_notifications()

    def on_execute(self):
        binary_task.LastBinaryTaskRelease.on_execute(self)

        repo_pathes = {
            resource.resource_name: sdk2.ResourceData(resource).path.as_posix()
            for resource in self.Parameters.resources
            if is_rasp_dict(resource.__class__)
        }

        log.info(repo_pathes)

        from travel.rasp.rasp_data.resource_checker import DataProvider, run
        data_provider = DataProvider(repo_pathes)
        checks_result = run(data_provider, log)
        if checks_result:
            self.set_info('Failed checks:\n{}'.format('\n'.join(checks_result)))
            raise common.errors.TaskFailure()

        self.set_info('Nice resources! All checks are done!')
