# -*- coding: utf-8 -*-
from __future__ import absolute_import, division, print_function, unicode_literals

import os
from operator import methodcaller
from os.path import join

import sandbox.common.types.client as ctc
from sandbox import sdk2
from sandbox.projects.common.ya_deploy.release_integration import ReleaseToNannyAndYaDeployTask2
from sandbox.projects.rasp.resource_types import RaspDBDumpResource
from sandbox.projects.rasp.utils.email_notifications import EmailNotificationMixin, use_email_notification_params
from sandbox.projects.rasp.utils.juggler import JugglerNotificationMixin, use_juggler_notification_params


class DumpMysqlTask(ReleaseToNannyAndYaDeployTask2, sdk2.Task, EmailNotificationMixin, JugglerNotificationMixin):
    class Requirements(sdk2.Task.Requirements):
        client_tags = ctc.Tag.LXC
        ram = 3 * 1024
        tasks_resource = sdk2.Task.Requirements.tasks_resource(default=2365805423)

    class Parameters(sdk2.Task.Parameters):

        host = sdk2.parameters.String('Host name', required=True)
        user = sdk2.parameters.String('User', default='root', required=True)
        database_name = sdk2.parameters.String('Database name', default='rasp', required=True)
        password_vault_name = sdk2.parameters.String('Password vault name', required=True)
        dump_directory = sdk2.parameters.String('Dump directory', default='dumps', required=True)
        environment = sdk2.parameters.String('Environment', default='dev', required=True)

        DEFAULTS = {
            'Currency': 100,
            'RThread': 200000,
            'RTStation': 300000,
            'Settlement': 70000,
            'Station': 200000,
            'Station2Settlement': 1000,
            'Supplier': 1000,
            'ThreadTariff': 200000,
        }

        (currency_count, rthread_count, rtstation_count, settlement_count, station_count,
         station2settlement_count, supplier_count, threadtariff_count) = (
            sdk2.parameters.Integer(
                'Approximated amount of {} objects in file'.format(name),
                default=default,
                required=True,
            )
            for name, default in sorted(DEFAULTS.items(), key=lambda pair: pair[0].lower())
        )

        _email_notification_params = use_email_notification_params()
        _juggler_notification_params = use_juggler_notification_params()

    @staticmethod
    def setup_paths(*paths):
        for p in paths:
            os.makedirs(p)

    def on_save(self):
        super(DumpMysqlTask, self).on_save()
        self.add_email_notifications()
        self.add_juggler_notifications(environment=self.Parameters.environment)

    def on_execute(self):
        from travel.rasp.mysql_dumper.lib.normalizer import run_normalize
        from travel.rasp.mysql_dumper.lib.dumper_runner import run_dump

        dump_directory = sdk2.ResourceData(RaspDBDumpResource(
            self, 'Dump directory', self.Parameters.dump_directory, environment=self.Parameters.environment
        ))
        dump_list = map(methodcaller('lower'), self.Parameters.DEFAULTS.keys())

        def get_dump_directory(name):
            return join(str(dump_directory.path), name)

        directories = {
            name: get_dump_directory(name)
            for name in dump_list
        }

        directories['precache'] = get_dump_directory('precache')

        self.setup_paths(*directories.values())
        db_password = sdk2.Vault.data(self.Parameters.password_vault_name)

        run_normalize(
            host=self.Parameters.host,
            user=self.Parameters.user,
            password=db_password,
            db=self.Parameters.database_name,
            directory=directories['precache'],
        )
        for name in dump_list:
            run_dump(
                host=self.Parameters.host,
                user=self.Parameters.user,
                password=db_password,
                db=self.Parameters.database_name,
                precache_path=join(directories['precache'], 'precache'),
                file_by_type={
                    name: directories[name]
                },
            )
