# -*- coding: utf-8 -*-
from __future__ import absolute_import, division, print_function, unicode_literals

import logging
import shutil
import tarfile
from pathlib2 import Path

from sandbox import sdk2
from sandbox.common.errors import TaskFailure
from sandbox.projects.Travel.resources.dicts import TRAVEL_RASP_COMMON_DICTS_BUNDLE
from sandbox.projects.rasp.resource_types import DumpRaspDataResource
from sandbox.projects.rasp.utils.email_notifications import (
    EmailNotificationMixin,
    use_email_notification_params,
    RASP_GROUP,
    AVIA_GROUP,
)
from sandbox.projects.rasp.utils.geobase import GeobaseMixin, use_geobase_params
from sandbox.projects.rasp.utils.juggler import JugglerNotificationMixin, use_juggler_notification_params
from sandbox.projects.common.ya_deploy.release_integration import ReleaseToNannyAndYaDeployTask2
from sandbox.sdk2.helpers import subprocess as sp


log = logging.getLogger()


class DumpRaspData(
    ReleaseToNannyAndYaDeployTask2,
    sdk2.Task,
    EmailNotificationMixin,
    JugglerNotificationMixin,
    GeobaseMixin
):
    DUMPER_PATH = 'dumper/dumper'
    OUTPUT_PATH = './output'
    COMMON_DICTS_PATH = './output/common_dicts'
    COMMON_DICTS_ARCHIVE_NAME = 'common_dicts'

    class Requirements(sdk2.Task.Requirements):
        ram = 3 * 1024

    class Parameters(sdk2.Task.Parameters):
        kill_timeout = 600

        with sdk2.parameters.Group('General') as general:
            host = sdk2.parameters.String('Host name', default='localhost', required=True)
            user = sdk2.parameters.String('User', default='root', required=True)
            environment = sdk2.parameters.String('Environment type', required=True)
            database_name = sdk2.parameters.String('Database name', default='rasp', required=True)
            password_vault_name = sdk2.parameters.String('Password vault name', required=True)
            tanker_auth_token_vault_name = sdk2.parameters.String('Tanker auth token vault name', required=True)
            build_common_dicts_archive = sdk2.parameters.Bool('Build common dicts archive', default=False)

        _email_notification_params = use_email_notification_params()
        _juggler_notification_params = use_juggler_notification_params()
        _geobase_params = use_geobase_params()

    def on_save(self):
        super(DumpRaspData, self).on_save()
        self.add_email_notifications(RASP_GROUP)
        self.add_email_notifications(AVIA_GROUP)
        self.add_juggler_notifications(environment=self.Parameters.environment)

    @classmethod
    def _find_last_resource(cls, attrs=None):
        count = DumpRaspDataResource.find(attrs=attrs).count
        resource = DumpRaspDataResource.find(attrs=attrs).order(DumpRaspDataResource.id).offset(count - 1).first()
        log.info('Found resource {}'.format(resource.id))

        return resource

    @classmethod
    def _extract_from_resource(cls, resource, path_to_member):
        data = sdk2.ResourceData(resource)
        data_path = data.path.absolute().as_posix()
        tar_file = tarfile.open(data_path)
        tar_file.extract(member=path_to_member)
        log.info('Extracted {} from {}'.format(path_to_member, data_path))

    def _run_dumper(self):
        db_password = sdk2.Vault.data(self.Parameters.password_vault_name)
        tanker_auth_token = sdk2.Vault.data(self.Parameters.tanker_auth_token_vault_name)
        dumper_path = './{}'.format(self.DUMPER_PATH)

        output = ('Running {path} --host {host} --user {user} --password {password} '
                  + '--name {name} --tankerAuthToken {token} --geobasePath {geobase} --out {out}')
        log.info(output.format(**{
            'path': dumper_path,
            'host': self.Parameters.host,
            'user': self.Parameters.user,
            'password': '***',
            'name': self.Parameters.database_name,
            'token': '***',
            'geobase': self.geobase_path,
            'out': self.OUTPUT_PATH,
        }))

        with sdk2.helpers.ProcessLog(self, logger="dumper_logger") as pl:
            cmd = [
                dumper_path,
                '--host', self.Parameters.host,
                '--user', self.Parameters.user,
                '--password', db_password,
                '--name', self.Parameters.database_name,
                '--tankerAuthToken', tanker_auth_token,
                '--geobasePath', self.geobase_path,
                '--out', self.OUTPUT_PATH,
            ]
            return_code = sp.Popen(cmd, stdout=pl.stdout, stderr=sp.STDOUT).wait()
        if return_code:
            raise TaskFailure('Dumper exit with return code {}'.format(return_code))

    def _upload_resource(self, output_file):
        output_file_absolute_path = output_file.absolute()
        log.info('Uploading {}'.format(output_file_absolute_path))

        base_name = output_file.name
        resource_class = self._get_resource_by_name(base_name)
        if resource_class is None:
            log.error("Can't find resource_class for resource name %s", base_name)
            return

        resource = resource_class(self, description=base_name, path=base_name.replace('.bin', '.data'), ttl='inf')
        resource_data = sdk2.ResourceData(resource)
        shutil.copy(
            src=output_file_absolute_path.as_posix(),
            dst=resource_data.path.absolute().as_posix(),
        )
        resource_data.ready()

    def _upload_common_dicts_archive(self, output_files):
        common_dicts_dir = Path(self.COMMON_DICTS_PATH)
        common_dicts_dir.mkdir(parents=True, exist_ok=True)
        log.info('Copying common dicts into %s', common_dicts_dir.absolute().as_posix())

        for output_file in output_files:
            try:
                output_file_absolute_path = output_file.absolute()
                base_name = output_file.name
                resource_class = self._get_resource_by_name(base_name)
                if resource_class is None:
                    log.error("Can't find resource_class for resource name %s", base_name)
                    continue

                if resource_class.is_common_dict:
                    log.info('Copying %s', output_file_absolute_path.as_posix())
                    new_filename = output_file_absolute_path.as_posix().replace('.bin', '.data')
                    shutil.copy(
                        src=output_file_absolute_path.as_posix(),
                        dst=new_filename,
                    )
                    shutil.move(
                        src=new_filename,
                        dst=common_dicts_dir.absolute().as_posix(),
                    )
            except Exception:
                log.exception('Error while adding %s to common dicts directory', output_file)
                raise

        try:
            log.info('Building archive')
            archive_path = shutil.make_archive(self.COMMON_DICTS_ARCHIVE_NAME, 'gztar', root_dir=self.COMMON_DICTS_PATH, base_dir='.')
            log.info('Archive has been built: %s', archive_path)
            archive_path = Path(archive_path)
            base_name = archive_path.absolute().name
            resource = TRAVEL_RASP_COMMON_DICTS_BUNDLE(self, description=base_name, path=base_name, ttl='inf')
            resource_data = sdk2.ResourceData(resource)
            log.info('Created common dicts resource data with path: %s', resource_data.path.absolute().as_posix())
            resource_data.ready()
        except Exception:
            log.exception('Error while uploading common dicts archive')
            raise

    @staticmethod
    def _load_resources():
        from sandbox.projects.Travel.resources import dicts

        result = {}
        for name, entry in dicts.__dict__.items():
            if dicts.is_rasp_dict(entry):
                result[entry.resource_name] = entry
        return result

    def _get_resource_by_name(self, base_name):
        name, _ = base_name.split('.')
        if not hasattr(self, '_resources'):
            self._resources = self._load_resources()

        return self._resources.get(name)

    def on_execute(self):
        self._extract_from_resource(
            resource=self._find_last_resource({'resource_name': 'dumper'}),
            path_to_member=self.DUMPER_PATH,
        )

        self.download_geobase()

        output_dir = Path(self.OUTPUT_PATH)
        output_dir.mkdir(parents=True, exist_ok=True)
        log.info('Created directory {}'.format(output_dir.absolute()))

        self._run_dumper()

        output_files = [p for p in output_dir.iterdir() if p.is_file()]
        for output_file in output_files:
            try:
                self._upload_resource(output_file)
            except Exception:
                log.exception('Error while uploading %s', output_file)

        if self.Parameters.build_common_dicts_archive:
            self._upload_common_dicts_archive(output_files)

        log.info('Done')
