import json
import logging
import subprocess

from pathlib2 import Path
import requests

from sandbox import sdk2
from sandbox.projects.rasp.resource_types import DumpTrainDataResource
from sandbox.projects.rasp.utils.email_notifications import EmailNotificationMixin, use_email_notification_params

log = logging.getLogger()


class DumpTrainData(sdk2.Task, EmailNotificationMixin):
    EXECUTABLE_NAME = 'dumper'
    POSTGRESQL_CERT_URL = 'https://crls.yandex.net/allCAs.pem'
    SECRET_BY_ENV = {
        'testing': 'sec-01dzv8rzv6xejtezj3jq0ks7zp',
        'production': 'sec-01e0d8g8g1ddafyg35n12wgzd7'
    }

    class Parameters(sdk2.Task.Parameters):
        description = 'Dump train data'
        kill_timeout = 600

        with sdk2.parameters.Group('General') as general:
            environment = sdk2.parameters.String('Environment type', default='testing', required=True)

        _email_notification_params = use_email_notification_params()

    @staticmethod
    def _find_last_resource(**attrs):
        last_resource = DumpTrainDataResource \
            .find(attrs=attrs) \
            .order(-DumpTrainDataResource.id) \
            .first()
        log.info('Found resource #{}'.format(last_resource.id))

        return last_resource

    @classmethod
    def _prepare_postgres_cert(cls):
        cert_path = Path.home() / '.postgresql/root.crt'
        if cert_path.exists():
            log.info('PostgreSQL CA cert already exists')
            return
        cert_path.parent.mkdir()
        cert_path.touch(0o600)

        response = requests.get(cls.POSTGRESQL_CERT_URL, timeout=10)
        response.raise_for_status()
        with cert_path.open('w', encoding='utf-8') as file:
            file.write(response.text)

        log.info('Downloaded PostgreSQL CA cert to {}'.format(cert_path))

    def _get_secret(self):
        secret = sdk2.yav.Secret(self.SECRET_BY_ENV[self.Parameters.environment]).data()
        log.info('Last version of secret is {}'.format(secret.version))

        return secret

    def _run_dumper(self):
        last_resource = self._find_last_resource(resource_name='dumper')
        dumper_path = sdk2.ResourceData(last_resource).path / self.EXECUTABLE_NAME
        secret = self._get_secret()
        self._prepare_postgres_cert()

        log.info('Running {} with {} environment'.format(dumper_path, self.Parameters.environment))
        env_vars = {
            'YENV_TYPE': self.Parameters.environment,
            'SANDBOX_SECRETS_FROM_ENV': json.dumps(secret)
        }
        proc = subprocess.Popen(str(dumper_path), stdout=subprocess.PIPE, stderr=subprocess.PIPE, env=env_vars)
        stdout, stderr = proc.communicate()

        if stdout:
            log.debug(stdout)
        if stderr:
            log.error(stderr)
            raise Exception('Error during _run_dumper')

    def _upload_resource(self, filename):
        log.info('Uploading {}'.format(filename))

        base_name = filename.name
        resource = DumpTrainDataResource(self, 'Train data protobuf', base_name,
                                         resource_name=base_name, environment=self.Parameters.environment)
        resource_data = sdk2.ResourceData(resource)
        resource_data.ready()

        log.info('Uploaded {}'.format(filename))

    def on_execute(self):
        self._run_dumper()
        for resource_file in Path.cwd().glob('*.bin'):
            self._upload_resource(resource_file)

        log.info('Finished the task')

    def on_save(self):
        super(DumpTrainData, self).on_save()
        self.add_email_notifications()
