# -*- coding: utf-8 -*-
from __future__ import absolute_import, division, print_function, unicode_literals

import logging
from datetime import timedelta

from sandbox import common
from sandbox import sdk2
from sandbox.common.types import task as ctt
from sandbox.common.types.misc import NotExists

from sandbox.projects.rasp.CheckRaspDataResources import RaspCheckRaspDataResources
from sandbox.projects.rasp.DumpMysqlDb import DumpMysqlTask
from sandbox.projects.rasp.DumpRaspData import DumpRaspData
from sandbox.projects.rasp.qloud.resources_config import list_environments_for_resource_update
from sandbox.projects.rasp.qloud.UpdateResources import RaspQloudUpdateResources
from sandbox.projects.rasp.resource_types import RaspDBDumpResource
from sandbox.projects.rasp.utils.email_notifications import EmailNotificationMixin, use_email_notification_params
from sandbox.projects.rasp.utils.juggler import JugglerNotificationMixin, use_juggler_notification_params
from sandbox.projects.Travel.resources import dicts


MYSQL_PASSWORD_VAULT_NAME_BY_ENV = {
    'production': 'rasp_db_mdb_password',
    'testing': 'rasp_test_db_mdb_password'
}


class RaspBuildRaspDataAndUpdate(sdk2.Task, EmailNotificationMixin, JugglerNotificationMixin):
    """
    Пересоздаем выгрузку базы и выкладываем изменившийся данные в продакшн или тестинг.

    Когда вызывается таск:
    Во время всех пересчетов
    """
    class Parameters(sdk2.Task.Parameters):
        kill_timeout = timedelta(hours=1).seconds

        with sdk2.parameters.Group('Rasp data maker') as mysql_dumper_block:
            rasp_data_maker__maker_host = sdk2.parameters.String(
                'MySQL host name',
                required=True
            )
            rasp_data_maker__maker_user = sdk2.parameters.String(
                'MySQL user',
                default='root',
                required=True
            )
            rasp_data_maker__maker_database_name = sdk2.parameters.String(
                'MySQL database name',
                default='rasp',
                required=True,
            )
            rasp_data_maker__maker_environment = sdk2.parameters.String(
                'Environment type',
                default='dev',
                required=True,
            )
            rasp_data_maker__maker_tanker_auth_token_vault_name = sdk2.parameters.String(
                'Tanker auth token vault name',
                default='rasp_tanker_auth_token',
                required=True,
            )
            rasp_data_maker__maker_build_common_dicts_archive = sdk2.parameters.Bool(
                'Build common dicts archive',
                default=False,
            )

        with sdk2.parameters.Group('Qloud resource updater') as _qloud_group:
            qloud_resource_updater__token_owner = sdk2.parameters.String(
                'Qloud secret vault owner',
                default='RASP',
                required=True
            )
            qloud_resource_updater__token_name = sdk2.parameters.String(
                'Qloud secret vault name',
                default='rasp-qloud-oauth',
                required=True
            )
            qloud_resource_updater__environments = sdk2.parameters.List(
                'Qloud environments to override rasp.qloud.resources_config',
            )

        with sdk2.parameters.Group('Release control') as _release_control:
            release_on_ready = sdk2.parameters.Bool(
                'Release dicts when task is ready',
                default=False
            )

        _email_notification_params = use_email_notification_params()
        _juggler_notification_params = use_juggler_notification_params()

    def on_enqueue(self):
        environment = self.Parameters.rasp_data_maker__maker_environment
        if environment != 'dev':
            name = '{}_for_{}'.format(self.type.name, environment)
            self.Requirements.semaphores = ctt.Semaphores(
                acquires=[
                    ctt.Semaphores.Acquire(name=name, capacity=1)
                ]
            )

    def __setup(self):
        context = self.Context
        self.__save_state('__make_rasp_data_resource')
        context.making_rasp_data_resource_attemps = 3
        context.checking_rasp_data_resource_attempts = 1
        context.updating_rasp_data_resource_attempts = 3
        self.__process_state()

    def __make_rasp_data_resource(self):
        environment = self.Parameters.rasp_data_maker__maker_environment
        mysql_vault_name = MYSQL_PASSWORD_VAULT_NAME_BY_ENV.get(environment)

        tasks = [
            DumpMysqlTask(
                self,
                owner=self.Parameters.owner,
                priority=self.Parameters.priority,
                description='Make legacy rasp data resource',

                host=self.Parameters.rasp_data_maker__maker_host,
                user=self.Parameters.rasp_data_maker__maker_user,
                environment=environment,
                database_name=self.Parameters.rasp_data_maker__maker_database_name,
                password_vault_name=mysql_vault_name,

                enable_email_notifications=False
            ),
            DumpRaspData(
                self,
                owner=self.Parameters.owner,
                priority=self.Parameters.priority,
                description='Make rasp data resource',

                host=self.Parameters.rasp_data_maker__maker_host,
                user=self.Parameters.rasp_data_maker__maker_user,
                environment=environment,
                database_name=self.Parameters.rasp_data_maker__maker_database_name,
                password_vault_name=mysql_vault_name,
                tanker_auth_token_vault_name=self.Parameters.rasp_data_maker__maker_tanker_auth_token_vault_name,
                build_common_dicts_archive=self.Parameters.rasp_data_maker__maker_build_common_dicts_archive,

                enable_email_notifications=True
            )
        ]
        for task in tasks:
            task.enqueue()

        rasp_data_resource_task_ids = [task.id for task in tasks]
        self.Context.rasp_data_resource_task_ids = self.Context.waited_task_ids = rasp_data_resource_task_ids
        self.__save_state('__check_making_rasp_data_resource')

        raise sdk2.WaitTask(rasp_data_resource_task_ids, ctt.Status.Group.FINISH | ctt.Status.Group.BREAK)

    def __check_making_rasp_data_resource(self):
        self.__check_subtasks_status(
            ok_state='__validate_rasp_data_resource',
            error_state='__make_rasp_data_resource',
            attempts_key='making_rasp_data_resource_attemps',
            delay=timedelta(minutes=1)
        )

    def _get_resources(self):
        return (
            sdk2.Resource.find(
                task_id=self.Context.rasp_data_resource_task_ids,
                type=RaspDBDumpResource
            ).first(),
        ) + tuple([
            resource for resource in sdk2.Resource.find(task_id=self.Context.rasp_data_resource_task_ids).limit(100)
            if dicts.is_rasp_dict(resource.__class__)
        ])

    def __validate_rasp_data_resource(self):
        rasp_data_resources = self._get_resources()

        tasks = [
            RaspCheckRaspDataResources(
                self,
                owner=self.Parameters.owner,
                priority=self.Parameters.priority,
                description='Check resources after DumpRaspData',

                resources=rasp_data_resources
            )
        ]

        for t in tasks:
            t.enqueue()

        self.Context.waited_task_ids = [task.id for task in tasks]
        self.__save_state('__check_validating_rasp_data_resources')

        raise sdk2.WaitTask(tasks, ctt.Status.Group.FINISH | ctt.Status.Group.BREAK)

    def __check_validating_rasp_data_resources(self):
        self.__check_subtasks_status(
            ok_state='__update_rasp_data_resource',
            error_state='__validate_rasp_data_resource',
            attempts_key='checking_rasp_data_resource_attempts',
            delay=timedelta(minutes=1)
        )

    def __get_release_type(self):
        if self.Parameters.rasp_data_maker__maker_environment in ('stable', 'production'):
            return ctt.ReleaseStatus.STABLE
        return ctt.ReleaseStatus.TESTING

    def __update_rasp_data_resource(self):
        rasp_data_resources = self._get_resources()

        self.Context.rasp_data_resources_ids = [resource.id for resource in rasp_data_resources]

        qloud_environments = filter(None, (
            qloud_environment.strip()
            for qloud_environment in self.Parameters.qloud_resource_updater__environments
        ))
        if not qloud_environments:
            qloud_environments = list_environments_for_resource_update(
                self.Parameters.rasp_data_maker__maker_environment
            )

        logging.debug('Updating qloud environments: %r', qloud_environments)

        tasks = [
            RaspQloudUpdateResources(
                self,
                owner=self.Parameters.owner,
                priority=self.Parameters.priority,
                description='Update resource',

                token_owner=self.Parameters.qloud_resource_updater__token_owner,
                token_name=self.Parameters.qloud_resource_updater__token_name,

                resources=rasp_data_resources,
                qloud_environment=qloud_environment,

                enable_email_notifications=False
            )
            for qloud_environment in qloud_environments
        ]

        for t in tasks:
            t.enqueue()
        task_ids = [t.id for t in tasks]

        self.Context.waited_task_ids = task_ids
        self.__save_state('__check_updating_rasp_data_resource')

        raise sdk2.WaitTask(task_ids, ctt.Status.Group.FINISH | ctt.Status.Group.BREAK)

    def __check_updating_rasp_data_resource(self):
        self.__check_subtasks_status(
            ok_state='__finalize',
            error_state='__update_rasp_data_resource',
            attempts_key='updating_rasp_data_resource_attempts',
            delay=timedelta(minutes=1)
        )

    def __finalize(self):
        logging.info('We have done it')

    def __process_unknown_handler(self):
        raise common.errors.TaskFailure('We do not have handler for: {}'.format(self.Context.state))

    def __save_state(self, next_state):
        current_state = self.Context.state
        logging.info('Move from [%s] to [%s]: %r', current_state, next_state, self.Context)
        self.Context.state = next_state

    def __process_state(self):
        current_state = self.Context.state
        logging.info('Current state: %s', current_state)

        if current_state is NotExists:
            logging.info('Process none state')
            self.__save_state('__setup')
            self.__process_state()
            return

        handler = getattr(self, '_RaspBuildRaspDataAndUpdate{}'.format(current_state), None)  # FIXME
        if handler is None:
            logging.info('Process none handler')
            self.__save_state('__process_unknown_handler')
            self.__process_state()
            return

        handler()

    def __decrement_attempt(self, attempts_key):
        attempts = getattr(self.Context, attempts_key)
        attempts -= 1
        setattr(self.Context, attempts_key, attempts)

        if attempts <= 0:
            raise common.errors.TaskFailure(
                'We can not have an attempt anymore for state: {}'.format(self.Context.state)
            )
        else:
            logging.info('We have %d attempts for %s key', attempts, attempts_key)

    def __check_subtasks_status(self, ok_state, error_state, attempts_key, delay):
        task_ids = self.Context.waited_task_ids
        tasks = [
           sdk2.Task.find(id=task_id).first()
           for task_id in task_ids
        ]

        if any(t is None for t in tasks):
            raise common.errors.TaskFailure('We can not find one of task by id: {!r}'.format(task_ids))

        broken_tasks = [t for t in tasks if t.status != ctt.Status.SUCCESS]
        if broken_tasks:
            self.__decrement_attempt(attempts_key)
            self.__save_state(error_state)
            raise sdk2.WaitTime(delay.seconds)

        self.__save_state(ok_state)
        self.__process_state()

    def on_execute(self):
        self.__process_state()

    def __release_rasp_data_resource(self):
        tasks = [
            sdk2.Task.find(id=self.Context.rasp_data_resource_task_ids, type=DumpRaspData).first(),
            sdk2.Task.find(id=self.Context.rasp_data_resource_task_ids, type=DumpMysqlTask).first(),
        ]
        for t in tasks:
            logging.info('create_release for {}'.format(t.id))
            release_id = self.server.release(
                task_id=t.id,
                type=self.__get_release_type(),
                subject='Releasing {} created on {}'.format(t.id, t.created)
            )

            if release_id is None:
                raise common.errors.TaskFailure('Release for task {} failed!'.format(t.id))
            else:
                logging.info('Released: {}'.format(release_id))

    def on_success(self, prev_status):
        super(RaspBuildRaspDataAndUpdate, self).on_success(prev_status)
        if self.__get_release_type() == ctt.ReleaseStatus.TESTING or self.Parameters.release_on_ready:
            self.__release_rasp_data_resource()

    def on_save(self):
        super(RaspBuildRaspDataAndUpdate, self).on_save()
        self.add_email_notifications()
        self.add_juggler_notifications(environment=self.Parameters.rasp_data_maker__maker_environment)
