# encoding: utf-8
from __future__ import absolute_import, division, print_function, unicode_literals

from sandbox import sdk2
from sandbox.common.errors import TaskFailure
from sandbox.common.types.task import Status
from sandbox.projects.BuildDockerImageFromGit import BuildDockerImageFromGit
from sandbox.projects.rasp.qloud.RestartEnvironment import RaspQloudRestartEnvironment
from sandbox.projects.rasp.utils.email_notifications import EmailNotificationMixin, use_email_notification_params


class RaspRebuildDocker(sdk2.Task, EmailNotificationMixin):
    class Parameters(sdk2.Task.Parameters):
        kill_timeout = 3600

        with sdk2.parameters.Group('Git parameters') as git_block:
            repository = sdk2.parameters.String('Repository', required=True)
            branch = sdk2.parameters.String('Branch', default='master')
            path_to_dockerfile = sdk2.parameters.String('Path to dockerfile inside the repository', default='.')
            ssh_vault_name = sdk2.parameters.String('Vault item with ssh key for git access')
            ssh_vault_owner = sdk2.parameters.String('Vault item owner')

        with sdk2.parameters.Group('Registry parameters') as docker_block:
            registry_tag = sdk2.parameters.String('Tag to publish image with (registry.yandex.net/<this tag>)')
            oauth_vault_name = sdk2.parameters.String(
                'Vault item with oauth token for '
                'registry.yandex.net (vault item name)'
            )
            oauth_vault_owner = sdk2.parameters.String('Vault item owner')
            registry_login = sdk2.parameters.String('Yandex login to use with docker login')

        with sdk2.parameters.Group('Qloud Restart') as qloud_block:
            qloud_component_paths = sdk2.parameters.String('Qloud components to restart, comma separated, '
                                                           'use asteriks for wildcards, for example: '
                                                           'rasp.blablacar.testing.backend,rasp.blablacar.static.*')

        _email_notification_params = use_email_notification_params()

    def check_subtasks(self):
        if not all(task.status == Status.SUCCESS for task in self.find()):
            raise TaskFailure('Some of child tasks failed')

    def on_execute(self):
        with self.memoize_stage.first_step:
            raise sdk2.WaitTask(
                [
                    BuildDockerImageFromGit(
                        self,
                        description='Updating docker image from TeamCity.',
                        repository=self.Parameters.repository,
                        branch=self.Parameters.branch,
                        path_to_dockerfile=self.Parameters.path_to_dockerfile,
                        ssh_vault_name=self.Parameters.ssh_vault_name,
                        ssh_vault_owner=self.Parameters.ssh_vault_owner,

                        registry_tags=[self.Parameters.registry_tag],
                        oauth_vault_name=self.Parameters.oauth_vault_name,
                        oauth_vault_owner=self.Parameters.oauth_vault_owner,
                        registry_login=self.Parameters.registry_login,
                    ).enqueue(),
                ],
                Status.Group.FINISH + Status.Group.BREAK,
                wait_all=True,
            )
        self.check_subtasks()

        with self.memoize_stage.second_step:
            raise sdk2.WaitTask(
                [
                    RaspQloudRestartEnvironment(
                        self,
                        description='Updating qloud components. Triggered by TeamCity.',
                        qloud_component_path=component.strip(),
                        registry_tag=self.Parameters.registry_tag,
                        token_owner=self.Parameters.ssh_vault_owner,
                        token_name=self.Parameters.oauth_vault_name,

                        enable_email_notifications=False
                    ).enqueue() for component in self.Parameters.qloud_component_paths.split(',')
                ],
                Status.Group.FINISH + Status.Group.BREAK,
                wait_all=True,
            )
        self.check_subtasks()

    def on_save(self):
        super(RaspRebuildDocker, self).on_save()
        self.add_email_notifications()
