# coding: utf8
from __future__ import absolute_import, division, print_function, unicode_literals

import logging
import os
import shutil

import sandbox.common.types.task as ctt
import sandbox.sdk2.helpers
from sandbox import common
from sandbox import sdk2
from sandbox.projects.rasp.qloud.RestartEnvironment import RaspQloudRestartEnvironment
from sandbox.projects.rasp.utils.changelog import generate_changelog
from sandbox.projects.rasp.utils.email_notifications import EmailNotificationMixin, use_email_notification_params
from sandbox.sdk2.helpers import subprocess
from sandbox.sdk2.vcs.git import Git


class RaspRelease(sdk2.Task, EmailNotificationMixin):
    """
    Релиз в Qloud с ченджлогом
    """

    checkout_path = 'src'

    class Parameters(sdk2.Task.Parameters):
        kill_timeout = 3600

        with sdk2.parameters.Group('Version parameters') as versions_block:
            version_tag_mark = sdk2.parameters.String('Version Tag Mark', required=False, default='')
            last_version_environment_name = sdk2.parameters.String(
                'Name of the environment, from which latest version tag will be taken',
                required=False, default='production')

        with sdk2.parameters.Group('Git parameters') as git_block:
            repository = sdk2.parameters.String('Repository', required=True)
            branch = sdk2.parameters.String('Branch or Tag', default='master')
            default_branch = sdk2.parameters.String('Default Branch, for cloning', default='master')
            ssh_vault_name = sdk2.parameters.String("Vault item with ssh key for git access")
            ssh_vault_owner = sdk2.parameters.String("Vault item owner")

        with sdk2.parameters.Group('Registry parameters') as docker_block:
            registry_url = sdk2.parameters.String('Registry Url of image, tags are taken from git'
                                                  ' (registry.yandex.net/<this url>:<tag>)')
            oauth_vault_name = sdk2.parameters.String(
                'Vault item with oauth token for '
                'registry.yandex.net (vault item name)'
            )
            oauth_vault_owner = sdk2.parameters.String('Vault item owner')

        with sdk2.parameters.Group('Qloud Restart') as qloud_block:
            qloud_component_paths = sdk2.parameters.String('Qloud components to restart, comma separated, '
                                                           'use asteriks for wildcards, for example: '
                                                           'rasp.blablacar.testing.backend,rasp.blablacar.static.*')

        _email_notification_params = use_email_notification_params()

    def gen_tag(self):
        version_tag_mark_params = (
            ['--version-tag-mark={}'.format(self.Parameters.version_tag_mark)] if self.Parameters.version_tag_mark else
            []
        )

        with sandbox.sdk2.helpers.ProcessLog(self, logger='get_tag'):
            current_tag = subprocess.check_output(
                ['python', './deploy_tools/versions.py', 'get-version'] + version_tag_mark_params,
                cwd=self.checkout_path,
                timeout=120
            ).strip()
            if not current_tag:
                raise common.errors.TaskFailure('Fail getting current commit tag')
            logging.info('Release version is %s', current_tag)
            registry_url = (
                self.Parameters.registry_url
                if self.Parameters.registry_url.startswith('registry.yandex.net/')
                else 'registry.yandex.net/' + self.Parameters.registry_url
            )
            self.Context.registry_tag = registry_url + ':' + current_tag

    def git_clone(self):
        if os.path.exists(self.checkout_path):
            shutil.rmtree(self.checkout_path)
        git = Git(self.Parameters.repository)
        git.clone(self.checkout_path, self.Parameters.default_branch)
        # !Обязательно нужно фетчить теги, версия гита страрая(~1.7) и перепушивает теги без форса
        # Кроме того, это нужно для генерации changelog
        git.execute('fetch', '--tags', cwd=self.checkout_path)
        git.execute('checkout', self.Parameters.branch, cwd=self.checkout_path)
        git.execute('submodule', 'update', '--init', cwd=self.checkout_path)

    def check_subtasks(self):
        if not all(task.status == ctt.Status.SUCCESS for task in self.find()):
            raise common.errors.TaskFailure('Some of child tasks failed')

    def on_execute(self):
        components = [env.strip() for env in self.Parameters.qloud_component_paths.split(',') if env]
        version_tag_mark_params = (
            ['--version-tag-mark={}'.format(self.Parameters.version_tag_mark)] if self.Parameters.version_tag_mark else
            []
        )

        with self.memoize_stage.first_step:
            with sdk2.ssh.Key(self, self.Parameters.ssh_vault_owner, self.Parameters.ssh_vault_name):
                self.git_clone()
                self.gen_tag()
                self.Context.changelog = generate_changelog(
                    self.Parameters.oauth_vault_owner,
                    self.Parameters.oauth_vault_name,
                    components,
                    self.Parameters.last_version_environment_name,
                    version_tag_mark_params,
                    self.checkout_path
                )

        with self.memoize_stage.second_step:
            if not components:
                raise common.errors.TaskFailure('There are no components to restart.')
            else:
                logging.info('Restart components %r', components)
                raise sdk2.WaitTask(
                    [
                        RaspQloudRestartEnvironment(
                            self,
                            description='Updating qloud environment. Triggered by TeamCity.',
                            qloud_component_path=component,
                            comment=self.Context.changelog,
                            registry_tag=self.Context.registry_tag,
                            token_name=self.Parameters.oauth_vault_name,
                            token_owner=self.Parameters.ssh_vault_owner,

                            priority=self.Parameters.priority,

                            enable_email_notifications=False
                        ).enqueue() for component in components
                    ],
                    ctt.Status.Group.FINISH + ctt.Status.Group.BREAK,
                    wait_all=True,
                )

        if components:
            with self.memoize_stage.check_restart_task:
                self.check_subtasks()

    def on_save(self):
        super(RaspRelease, self).on_save()
        self.add_email_notifications()
