# -*- coding: utf-8 -*-
from __future__ import absolute_import, division, print_function, unicode_literals

import logging
import os
import tarfile

from sandbox import sdk2
from sandbox.common.errors import TaskFailure
from sandbox.common.types import resource as ctr
from sandbox.common.types import task as ctt
from sandbox.projects.common import binary_task
from sandbox.projects.Travel.resources import TRAVEL_TRAINS_TEMPLATES_DUMPER_PACKAGE
from sandbox.projects.rasp.utils.email_notifications import (
    EmailNotificationMixin,
    use_email_notification_params,
    TRAIN_GROUP,
)
from sandbox.sdk2 import yav
from sandbox.sdk2.helpers import subprocess as sp

log = logging.getLogger(__name__)


class TrainsDumpTemplates(binary_task.LastBinaryTaskRelease, sdk2.Task, EmailNotificationMixin):
    BINARY_DIR = 'cmd/templates_dumper/'
    BINARY_PATH = BINARY_DIR + 'templates_dumper'
    CONFIG_PATH_TEMPLATE = 'docker/templates_dumper/config.{}.yaml'
    ENVIRONMENT_TO_RELEASED_MAP = {'testing': 'testing', 'production': 'stable'}
    YAV_TRAINS_COMMON_TESTING_SECRET = 'sec-01f3td8gjk2espct0f2djg53sm'
    YAV_TRAINS_COMMON_PRODUCTION_SECRET = 'sec-01f3tcs4pcx2q20ey3fsexg0wz'
    YAV_ROBOT_RASP_TOKENS_SECRET = 'sec-01cjvsaf2hrdkfdr97c30dv3t4'
    YAV_TANKER_OAUTH_TOKEN_KEY = 'tanker-oauth-token'
    YAV_S3_SECRET_KEY = 's3-trains-common-secret-key'
    YAV_S3_KEY_ID_KEY = 's3-trains-common-key-id'

    class Requirements(sdk2.Task.Requirements):
        cores = 1
        ram = 1 * 1024

        class Caches(sdk2.Requirements.Caches):  # enable multislot https://wiki.yandex-team.ru/sandbox/clients/#client-tags-multislot
            pass  # means that task do not use any shared caches

    class Parameters(sdk2.Task.Parameters):
        kill_timeout = 300
        description = 'Dump train templates from tanker to s3.'
        with sdk2.parameters.RadioGroup('Environment', required=True) as environment:
            environment.values['testing'] = environment.Value(value='testing', default=True)
            environment.values['production'] = environment.Value(value='production')
        templates_dumper = sdk2.parameters.Resource(
            'Templates dumper tarball resource. Latest if empty',
            default=None
        )
        ext_params = binary_task.binary_release_parameters(stable=True)

        _email_notification_params = use_email_notification_params()

    def on_execute(self):
        process_env = os.environ.copy()
        common_secret_id = self.YAV_TRAINS_COMMON_TESTING_SECRET
        if self.Parameters.environment == 'production':
            common_secret_id = self.YAV_TRAINS_COMMON_PRODUCTION_SECRET
        common_secret_data = yav.Secret(common_secret_id).data()
        robot_secret_data = yav.Secret(self.YAV_ROBOT_RASP_TOKENS_SECRET).data()
        process_env['TANKER_OAUTH_TOKEN'] = robot_secret_data[self.YAV_TANKER_OAUTH_TOKEN_KEY]
        process_env['SEO_S3_ACCESS_KEY'] = common_secret_data[self.YAV_S3_KEY_ID_KEY]
        process_env['SEO_S3_SECRET'] = common_secret_data[self.YAV_S3_SECRET_KEY]
        process_env_open = {}
        process_env_open['YENV_TYPE'] = self.Parameters.environment
        process_env_open['CONFIG_PATH'] = self.CONFIG_PATH_TEMPLATE.format(self.Parameters.environment)

        templates_dumper_resource = self.Parameters.templates_dumper
        if not templates_dumper_resource:
            templates_dumper_resource = self._find_last_resource()
        self._extract_all_from_resource(resource=templates_dumper_resource)

        log.info(process_env_open)
        process_env.update(process_env_open)
        binary_path = './{}'.format(self.BINARY_PATH)
        with sdk2.helpers.ProcessLog(self, logger="dumper_logger") as pl:
            dumper_out_log_path = pl.stdout.path
            return_code = sp.Popen([binary_path], stdout=pl.stdout, stderr=pl.stdout, env=process_env).wait()

        with open(str(dumper_out_log_path), 'r') as f:
            dumper_out = f.read()
        self.set_info(dumper_out)

        if return_code:
            raise TaskFailure('Dumper exit with return code {}'.format(return_code))

        logging.info('Done')

    def on_save(self):
        super(TrainsDumpTemplates, self).on_save()
        self.add_email_notifications(notifications_group=TRAIN_GROUP)

    @property
    def binary_executor_query(self):
        return {
            "attrs": {"task_type": "TRAINS_DUMP_TEMPLATES", "released": self.Parameters.binary_executor_release_type},
            "state": [ctr.State.READY]
        }

    def _find_last_resource(self):
        resource_class = TRAVEL_TRAINS_TEMPLATES_DUMPER_PACKAGE
        env = self.Parameters.environment
        resource = resource_class.find(limit=1, state=ctr.State.READY, attrs={'released': ctt.ReleaseStatus.STABLE}).first()
        resource_testing = None
        if env == 'testing':
            resource_testing = resource_class.find(limit=1, state=ctr.State.READY, attrs={'released': ctt.ReleaseStatus.TESTING}).first()
        if resource_testing and resource and resource_testing.id > resource.id:
            resource = resource_testing
        log.info('Found {}: id {}'.format(
            resource_class.__name__,
            resource.id
        ))
        return resource

    @classmethod
    def _extract_all_from_resource(cls, resource):
        data = sdk2.ResourceData(resource)
        data_path = data.path.absolute().as_posix()
        tar_file = tarfile.open(data_path)
        tar_file.extractall()
        log.info('Extracted all from {}'.format(data_path))
