# coding: utf8
from __future__ import absolute_import, division, print_function, unicode_literals

import os

from sandbox import sdk2
from sandbox.projects.common.binary_task import LastBinaryTaskRelease, binary_release_parameters

from sandbox.projects.rasp.utils.email_notifications import EmailNotificationMixin, use_email_notification_params
from sandbox.projects.rasp.utils.geobase import GeobaseMixin, use_geobase_params
from sandbox.projects.rasp.utils.juggler import JugglerNotificationMixin, use_juggler_notification_params
from sandbox.projects.rasp.utils.rasp_scripts import RaspScriptsTask, use_rasp_scripts_params
from sandbox.projects.rasp.utils.solomon import SolomonProgressMixin


class TrainsGenerateCanonical(
    sdk2.Task,
    RaspScriptsTask,
    LastBinaryTaskRelease,
    SolomonProgressMixin,
    JugglerNotificationMixin,
    EmailNotificationMixin,
    GeobaseMixin
):
    class Requirements(sdk2.Task.Requirements):
        cores = 1
        ram = 1024 * 10

        class Caches(sdk2.Requirements.Caches):  # https://wiki.yandex-team.ru/sandbox/clients/#client-tags-multislot
            pass  # task does not use any shared caches

    class Parameters(sdk2.Task.Parameters):
        kill_timeout = 3600
        ext_params = binary_release_parameters(stable=True)

        canonical_yt_table = sdk2.parameters.String('Canonical yt table', default=None)

        _rasp_scripts_params = use_rasp_scripts_params()
        _email_notification_params = use_email_notification_params()
        _juggler_notification_params = use_juggler_notification_params()
        _geobase_params = use_geobase_params()

    def set_task_environ(self):
        os.environ['RASP_GEOBASE_DATA_PATH'] = self.geobase_path
        os.environ['DJANGO_SETTINGS_MODULE'] = 'travel.rasp.trains.scripts.generate_canonical.settings'

    def on_save(self):
        LastBinaryTaskRelease.on_save(self)

        self.add_juggler_notifications(environment=self.Parameters.env_type)
        self.add_email_notifications()

    def on_execute(self):
        LastBinaryTaskRelease.on_execute(self)

        self.download_geobase()
        self.set_common_environ()

        from travel.rasp.trains.scripts.generate_canonical.generate_canonical import generate_canonical

        with self.report_progress('generate_canonical', self.Parameters.env_type):
            generate_canonical(canonical_yt_table=self.Parameters.canonical_yt_table)
