# coding: utf8
from __future__ import absolute_import, division, print_function, unicode_literals

import os

from sandbox import sdk2
from sandbox.projects.common.binary_task import LastBinaryTaskRelease, binary_release_parameters

from sandbox.projects.rasp.rasp_scripts.RaspScriptsUpdateQloud import RaspScriptsUpdateQloudMixin
from sandbox.projects.rasp.resource_types import TrainCrossLinksErrorsProduction, TrainCrossLinksErrorsTesting
from sandbox.projects.rasp.utils.email_notifications import EmailNotificationMixin, use_email_notification_params
from sandbox.projects.rasp.utils.geobase import GeobaseMixin, use_geobase_params
from sandbox.projects.rasp.utils.juggler import JugglerNotificationMixin, use_juggler_notification_params
from sandbox.projects.rasp.utils.rasp_scripts import RaspScriptsTask, use_rasp_scripts_params
from sandbox.projects.rasp.utils.solomon import SolomonProgressMixin


TRAIN_CROSSLINKS_ERRORS_RES = {
    'testing': TrainCrossLinksErrorsTesting,
    'production': TrainCrossLinksErrorsProduction
}


class TrainsGenerateCrossLinks(
    sdk2.Task,
    RaspScriptsTask,
    RaspScriptsUpdateQloudMixin,
    LastBinaryTaskRelease,
    SolomonProgressMixin,
    JugglerNotificationMixin,
    EmailNotificationMixin,
    GeobaseMixin
):
    class Requirements(sdk2.Task.Requirements):
        cores = 1
        ram = 1024 * 10

        class Caches(sdk2.Requirements.Caches):  # https://wiki.yandex-team.ru/sandbox/clients/#client-tags-multislot
            pass  # task does not use any shared caches

    class Parameters(sdk2.Task.Parameters):
        kill_timeout = 3600 * 2
        ext_params = binary_release_parameters(stable=True)

        links_count = sdk2.parameters.Integer('Links count', required=True, default=6)
        unviewed_links_count = sdk2.parameters.Integer('Unviewed links count', required=True, default=2)
        proximity_relevance = sdk2.parameters.Float('Proximity relevance', required=True, default=1.)
        distance_additive = sdk2.parameters.Float('Distance additive', required=True, default=0.1)
        stat_days_period = sdk2.parameters.Integer('Stat days period', required=True, default=30)

        crosslinks_yt_table = sdk2.parameters.String('Crosslinks yt table', default=None)

        _rasp_scripts_params = use_rasp_scripts_params()
        _email_notification_params = use_email_notification_params()
        _juggler_notification_params = use_juggler_notification_params()
        _geobase_params = use_geobase_params()

    def set_task_environ(self):
        os.environ['RASP_GEOBASE_DATA_PATH'] = self.geobase_path
        os.environ['DJANGO_SETTINGS_MODULE'] = 'travel.rasp.trains.scripts.generate_crosslinks.settings'

    def on_save(self):
        LastBinaryTaskRelease.on_save(self)

        if self.Parameters.links_count <= self.Parameters.unviewed_links_count:
            raise Exception('links_count <= unviewed_links_count')

        self.add_juggler_notifications(environment=self.Parameters.env_type)
        self.add_email_notifications()

    def on_execute(self):
        LastBinaryTaskRelease.on_execute(self)

        self.download_geobase()
        self.set_common_environ()

        from travel.rasp.trains.scripts.generate_crosslinks.generate_crosslinks import generate_links, ERRORS_FILE_NAME

        with self.report_progress('generate_crosslinks', self.Parameters.env_type):
            try:
                generate_links(
                    links_count=self.Parameters.links_count,
                    unviewed_links_count=self.Parameters.unviewed_links_count,
                    proximity_relevance=self.Parameters.proximity_relevance,
                    distance_additive=self.Parameters.distance_additive,
                    stat_days_period=self.Parameters.stat_days_period,
                    crosslinks_yt_table=self.Parameters.crosslinks_yt_table
                )
            except Exception:
                if os.path.exists(ERRORS_FILE_NAME):
                    errors_res = TRAIN_CROSSLINKS_ERRORS_RES.get(self.Parameters.env_type)
                    self._upload_resource(errors_res, ERRORS_FILE_NAME)
                raise
