# -*- coding: utf-8 -*-

from datetime import datetime, timedelta

import sandbox.common.types.client as ctc
from sandbox import sdk2
from sandbox.projects.rasp.utils import ISO_FORMAT
from sandbox.projects.rasp.utils.email_notifications import EmailNotificationMixin, use_email_notification_params
from sandbox.sandboxsdk import environments


class RaspAerotoStat(sdk2.Task, EmailNotificationMixin):
    class Requirements(sdk2.Task.Requirements):
        client_tags = ctc.Tag.LXC
        ram = 2 * 1024
        environments = [
            environments.PipEnvironment('python-statface-client', use_wheel=False),
        ]

    class Parameters(sdk2.Task.Parameters):
        with sdk2.parameters.Group('Aero Mongo parameters') as mongo_params:
            mongo_user = sdk2.parameters.String('Aero Mongo user', default='corroder', required=True)
            mongo_host = sdk2.parameters.String(
                'Aero Mongo database host',
                default='mongodb://sas-j9ll6pnz1yv019lm.db.yandex.net:27018',
                required=True
            )
            mongo_password_vault_name = sdk2.parameters.String('Aero Mongo pass vault name', required=True)

        with sdk2.parameters.Group('Stat parameters') as stat_params:
            stat_token_vault_name = sdk2.parameters.String('statface robot oauth token vault name',
                                                           default='robot-rasp-statbox-token', required=True)
            stat_host = sdk2.parameters.String('stat host', default='upload.stat.yandex-team.ru', required=True)
            stat_report = sdk2.parameters.String('stat report', default='Raspisanie/dapavlov/aeroorders', required=True)

        with sdk2.parameters.Group('Data query parameters') as data_params:
            dt_from = sdk2.parameters.String('ISO datetime from (default=dt_to - 1h)', required=False, default='')
            dt_to = sdk2.parameters.String('ISO datetime to (default=now)', required=False, default='')

        _email_notification_params = use_email_notification_params()

    @property
    def db(self):
        if getattr(self, '_db', None) is None:
            from pymongo import MongoClient
            db = MongoClient(self.Parameters.mongo_host).suburb
            mongo_pass = sdk2.Vault.data(self.Parameters.mongo_password_vault_name)
            db.authenticate(self.Parameters.mongo_user, mongo_pass)
            self._db = db
        return self._db

    @property
    def stat_report(self):
        if getattr(self, '_stat_report', None) is None:
            from statface_client import StatfaceClient
            stat_client_config = {
                'oauth_token': sdk2.Vault.data(self.Parameters.stat_token_vault_name),
                'host': self.Parameters.stat_host
            }
            stat_client = StatfaceClient(client_config=stat_client_config)
            report = stat_client.get_old_report(self.Parameters.stat_report)
            self._stat_report = report
        return self._stat_report

    def get_tickets_and_money_from_mongo(self, begin_date, end_date):
        pipeline = [
            {"$match": {
                "status": "done",
                "finished_at": {
                    '$gte': begin_date,
                    '$lt': end_date
                }
            }},
            {"$unwind": "$passengers"},
            {"$unwind": "$passengers.ticket"},
            {"$group": {
                "_id": "$tickets",
                "tickets_count": {"$sum": 1},
                "sum_price": {"$sum": "$passengers.ticket.ticket_price"}
            }}
        ]
        items = list(self.db.order.aggregate(pipeline))
        if items:
            result = items[0]
            return result
        else:
            return {'sum_price': 0, 'tickets_count': 0}

    def get_orders_from_mongo(self, begin_date, end_date):
        pipeline = [
            {"$match": {
                "status": "done",
                "finished_at": {
                    '$gte': begin_date,
                    '$lt': end_date
                }
            }},
            {"$group": {"_id": "_id",
                        "count": {"$sum": 1}}}
        ]
        items = list(self.db.order.aggregate(pipeline))
        if items:
            result = items[0]['count']
            return result
        else:
            return 0

    def get_failed_orders_from_mongo(self, begin_date, end_date):
        pipeline = [
            {"$match": {
                "status": {
                    "$in": [
                        'cancelled',
                        'payment_failed',
                        'confirm_failed',
                        'payment_failed',
                        'payment_outdated',
                        'start_payment_failed',
                    ]
                },
                "reserved_to": {
                    '$gte': begin_date,
                    '$lt': end_date
                }
            }},
            {"$group": {
                "_id": "_id",
                "count": {"$sum": 1}
            }}
        ]
        items = list(self.db.order.aggregate(pipeline))
        if items:
            result = items[0]['count']
            return result
        else:
            return 0

    def on_execute(self):
        if self.Parameters.dt_to:
            to_hour = datetime.strptime(self.Parameters.dt_to, ISO_FORMAT)
        else:
            to_hour = datetime.today()
        to_hour = to_hour.replace(minute=0, second=0, microsecond=0)
        if self.Parameters.dt_from:
            from_hour = datetime.strptime(self.Parameters.dt_from, ISO_FORMAT).replace(minute=0, second=0, microsecond=0)
        else:
            from_hour = to_hour - timedelta(hours=1)

        data = []
        hour = from_hour
        while hour <= to_hour:
            next_hour = hour + timedelta(hours=1)
            m_hour = hour - timedelta(hours=3)
            m_next_hour = next_hour - timedelta(hours=3)
            money_ticket = self.get_tickets_and_money_from_mongo(m_hour, m_next_hour)
            to_append = {
                'fielddate': str(hour),
                'tickets': money_ticket['tickets_count'],
                'orders': self.get_orders_from_mongo(m_hour, m_next_hour),
                'failed_orders': self.get_failed_orders_from_mongo(m_hour, m_next_hour),
                'price': money_ticket['sum_price'],
                'fee': money_ticket['tickets_count']*40}
            data.append(to_append)
            hour += timedelta(hours=1)

        self.stat_report.upload_data(scale='hourly', data=data)

    def on_save(self):
        super(RaspAerotoStat, self).on_save()
        self.add_email_notifications()
