# -*- coding: utf-8 -*-

from datetime import datetime, timedelta

from sandbox import sdk2
from sandbox.common.types.client import Tag
from sandbox.projects.rasp.utils import ISO_FORMAT
from sandbox.projects.rasp.utils.email_notifications import EmailNotificationMixin, use_email_notification_params
from sandbox.sandboxsdk import environments


class RaspRefundsToStat(sdk2.Task, EmailNotificationMixin):
    class Requirements(sdk2.Task.Requirements):
        client_tags = Tag.LXC
        ram = 2 * 1024
        environments = [
            environments.PipEnvironment('python-statface-client', use_wheel=False),
        ]

    class Parameters(sdk2.Task.Parameters):
        RZHD_STATUS_REFUNDED = 4
        RZHD_STATUS_PLACES_REFUNDED = 5

        with sdk2.parameters.Group('Stat parameters') as stat_params:
            stat_token_vault_name = sdk2.parameters.String('statface robot oauth token vault name',
                                                           default='robot-rasp-statbox-token', required=True)
            stat_host = sdk2.parameters.String('stat host', default='upload.stat.yandex-team.ru', required=True)
            stat_report = sdk2.parameters.String(
                'stat report',
                default='Raspisanie/dapavlov/refunds_by_platform',
                required=True
            )
        with sdk2.parameters.Group('Data query parameters') as data_params:
            date = sdk2.parameters.String('ISO datetime (default=now)', required=False, default='')

        _email_notification_params = use_email_notification_params()

    @property
    def db(self):
        if getattr(self, '_db', None) is None:
            from sandbox.projects.rasp.analytics.utils import get_train_purchase_db
            self._db = get_train_purchase_db()
        return self._db

    @property
    def stat_report(self):
        if getattr(self, '_stat_report', None) is None:
            from statface_client import StatfaceClient
            stat_client_config = {
                'oauth_token': sdk2.Vault.data(self.Parameters.stat_token_vault_name),
                'host': self.Parameters.stat_host
            }
            stat_client = StatfaceClient(client_config=stat_client_config)
            report = stat_client.get_old_report(self.Parameters.stat_report)
            self._stat_report = report
        return self._stat_report

    def get_refunded_orders_from_mongo(self, min_dt, max_dt):
        items = self.db.train_refund.find(
            {
                'created_at': {
                    '$gte': min_dt,
                    '$lt': max_dt,
                },
                'status': {
                    '$in': ['partner_refund_done', 'done']
                }
            },
            {
                'order_uid': '1',
            }
            )
        return [item['order_uid'] for item in items]

    def get_return_tickets_from_mongo(self, is_mobile, list_of_orders):
        pipeline = [
            {
                '$match': {
                    'status': 'done',
                    'user_info.is_mobile': is_mobile,
                    'uid': {
                        '$in': list_of_orders
                    },
                    'passengers.tickets.rzhd_status': {
                        '$in': [self.Parameters.RZHD_STATUS_REFUNDED, self.Parameters.RZHD_STATUS_PLACES_REFUNDED]
                    }
                }
            },
            {'$unwind': '$passengers'},
            {'$unwind': '$passengers.tickets'},
            {
                '$group': {
                    '_id': '$tickets',
                    'count': {'$sum': 1},
                    'sum_fee': {'$sum': '$passengers.tickets.payment.fee'},
                    'sum_amount': {'$sum': '$passengers.tickets.payment.amount'},
                    'sum_refund_amount': {'$sum': '$passengers.tickets.refund.amount'},
                    'sum_partner_fee': {'$sum': '$passengers.tickets.payment.partner_fee'}
                }
            }
        ]
        item = list(self.db.train_order.aggregate(pipeline))
        if item:
            return item[0]
        return {'sum_fee': 0, 'sum_amount': 0, 'sum_partner_fee': 0, 'count': 0, 'sum_refund_amount': 0}

    def get_mongo_info_dict_by_platform(self, source_device,
                                        is_mobile, current_dt, delta):
        max_dt = current_dt - timedelta(hours=3)
        min_dt = current_dt - delta - timedelta(hours=3)
        orders = self.get_refunded_orders_from_mongo(min_dt, max_dt)
        mongo_data = self.get_return_tickets_from_mongo(is_mobile, orders)

        stat_dict = {
            'fielddate': str(current_dt-delta),
            'ui': source_device,
            'tickets': mongo_data['count'],
            'price': mongo_data['sum_amount'],
            'refund_amount': mongo_data['sum_refund_amount'],
            'fee': mongo_data['sum_fee'],
            'partner_fee': mongo_data['sum_partner_fee']
        }
        return stat_dict

    def on_execute(self):
        data = []
        if self.Parameters.date:
            current_dt = datetime.strptime(self.Parameters.date, ISO_FORMAT)
        else:
            current_dt = datetime.today()
        current_dt = current_dt.replace(hour=0, minute=0, second=0, microsecond=0)
        delta = timedelta(days=1)

        desktop_data = self.get_mongo_info_dict_by_platform(
            source_device='desktop',
            is_mobile=False,
            current_dt=current_dt,
            delta=delta,
        )
        touch_data = self.get_mongo_info_dict_by_platform(
            source_device='touch',
            is_mobile=True,
            current_dt=current_dt,
            delta=delta,
        )
        data.append(desktop_data)
        data.append(touch_data)

        self.stat_report.upload_data(scale='daily', data=data)

    def on_save(self):
        super(RaspRefundsToStat, self).on_save()
        self.add_email_notifications()
