# -*- coding: utf-8 -*-
from __future__ import absolute_import, division, print_function, unicode_literals

import csv
import json
import logging

from os import environ
from pathlib2 import Path

import sandbox.common.types.misc as ctm
from sandbox import sdk2
from sandbox.projects.rasp.bus.BusBaseTask import BusBaseTaskAutoResource, BusAutoResourceParameters, SOLOMON_OFF_TAG
from sandbox.projects.rasp.bus.utils.resources import AutoMatcherResource
from sandbox.projects.rasp.bus.utils import get_data_resource
from sandbox.projects.Travel.resources import dicts


log = logging.getLogger(__name__)

RASP_DATA_CONFIG_JSON = """
{
    "ok": {
        "rasp_id": 542,
        "match_by": "code",
        "prefix": "s"
    },
    "etraffic": {
        "rasp_id": 114,
        "match_by": "etraffic_title"
    }
}
"""


def from_file_config(params):
    log.info('matching_strings: %s', params.from_file_data)
    config = {
        "matchings": params.from_file_data,
    }
    return config


def rasp_data_config(params):
    rasp_cfg = json.loads(params.rasp_data_config)
    return rasp_cfg


def dead_matchings_config(params):
    rasp_stations = get_data_resource(dicts.TRAVEL_DICT_RASP_STATION_PROD)
    if not rasp_stations:
        raise ValueError("cannot find resource TRAVEL_DICT_RASP_STATION_PROD")
    rasp_stations_data = sdk2.ResourceData(rasp_stations)

    rasp_settlements = get_data_resource(dicts.TRAVEL_DICT_RASP_SETTLEMENT_PROD)
    if not rasp_settlements:
        raise ValueError("cannot find resource TRAVEL_DICT_RASP_SETTLEMENT_PROD")
    rasp_settlements_data = sdk2.ResourceData(rasp_settlements)

    config = {
        "station_fn": str(rasp_stations_data.path.absolute()),
        "settlement_fn": str(rasp_settlements_data.path.absolute())
    }
    return config


class BusAutoMatcher(BusBaseTaskAutoResource):
    REPORT_PATH = 'buses_automatcher_resource'
    REPORT_FILENAME_TMPL = '{}.csv'
    RESOURCE_TTL = 30
    SCENARIOS = {
        'from_file': from_file_config,
        'etraffic_twin_stations': None,
        'unitiki_update': None,
        'rasp_data': rasp_data_config,
        'dead_matchings': dead_matchings_config,
    }

    class Requirements(sdk2.Task.Requirements):
        disk_space = 4024
        ram = 4048
        dns = ctm.DnsType.DNS64

    class Parameters(BusAutoResourceParameters):
        with sdk2.parameters.Group("Auto Matcher") as params:
            dry = sdk2.parameters.Bool('Dry', default=True, required=True)
            suppliers = sdk2.parameters.String("Suppliers (all if empty)")
            login = sdk2.parameters.String("Automatcher login", required=True, default='automatcher')
            run_all = sdk2.parameters.Bool('Run all pointkeys (matched too)', default=False, required=True)

        with sdk2.parameters.Group("Scenarios") as scns:
            from_file = sdk2.parameters.Bool('Load matchings from file', default=False)
            etraffic_twin_stations = sdk2.parameters.Bool('Etraffic twins stations', default=False)
            unitiki_update = sdk2.parameters.Bool('Apply unitiki changes', default=False)
            rasp_data = sdk2.parameters.Bool('Load from RASP data', default=False)
            dead_matchings = sdk2.parameters.Bool('Delete dead matchings', default=False)
            with rasp_data.value[True]:
                rasp_data_config = sdk2.parameters.String("RASP data config", multiline=True, required=True,
                                                          default=RASP_DATA_CONFIG_JSON)
            with from_file.value[True]:
                from_file_data = sdk2.parameters.String("From file data, csv format: name, point_id, matching",
                                                        multiline=True, required=True)

    def on_save(self):
        super(BusAutoMatcher, self).on_save()
        if SOLOMON_OFF_TAG not in self.Parameters.tags:
            self.Parameters.tags += [SOLOMON_OFF_TAG]

    def _get_scenarios_list(self):
        scenarios = []
        for (param, value) in self.Parameters:
            if param in self.SCENARIOS and value:
                scenarios.append(param)
            log.info('param: %s', param)
        return scenarios

    def _get_scenario_configs(self, scenarios):
        params = {}
        for scenario in scenarios:
            cfg_getter = self.SCENARIOS[scenario]
            if cfg_getter:
                params[scenario] = cfg_getter(self.Parameters)
        return params

    def on_execute(self):
        super(BusAutoMatcher, self).on_execute()
        suppliers_names = [supplier.strip() for supplier in self.Parameters.suppliers.split(',')] \
            if self.Parameters.suppliers else []

        scenarios = self._get_scenarios_list()
        log.info('scenarios: %s', scenarios)

        params = self._get_scenario_configs(scenarios)
        log.info('scenario params: %s', params)

        environ['YENV_TYPE'] = self.Parameters.environment
        environ['RASP_VAULT_OAUTH_TOKEN'] = sdk2.Vault.data(self.Parameters.sandbox_vault_owner, 'YAV_OAUTH_TOKEN')
        from travel.rasp.bus.scripts.automatcher import AutoMatcher

        auto_matcher = AutoMatcher(
            scenarios,
            suppliers_names,
            params,
            self.Parameters.login,
            self.Parameters.dry,
            self.Parameters.run_all,
        )
        auto_matcher.run()

        self._make_report(auto_matcher.get_stats())
        logging.info("Done!")

    def _make_report(self, automatcher_stats):
        resource_path = Path(self.REPORT_PATH)
        resource_path.mkdir(parents=True, exist_ok=True)
        report_resource = AutoMatcherResource(
            self, description='Automatcher reports', path=self.REPORT_PATH, ttl=self.RESOURCE_TTL
        )
        report_resource_data = sdk2.ResourceData(report_resource)
        for report_name, report_rows in automatcher_stats.items():
            report_filename = str(resource_path / self.REPORT_FILENAME_TMPL.format(report_name))
            with open(report_filename, 'w') as f:
                csvwriter = csv.writer(f, dialect='excel')
                csvwriter.writerows(report_rows)
            log.info('Report "%s" generated', report_name)
        report_resource_data.ready()
