# coding: utf8
from __future__ import absolute_import, division, print_function, unicode_literals

from sandbox import sdk2
from sandbox.common import errors
from sandbox.common.types import task as ctt
from sandbox.projects.Travel.tasks.tools.solomon_mixin import CommonTaskTrackerMixin


ENV_TESTING = 'testing'
ENV_PRODUCTION = 'production'

ENV_TO_CLUSTER_MAPPER = {
    ENV_TESTING: 'testing',
    ENV_PRODUCTION: 'production'
}

SOLOMON_OFF_TAG = 'NO_SOLOMON'

INHERITABLE_TAGS = {
    SOLOMON_OFF_TAG
}


class BusBaseParameters(sdk2.Task.Parameters):
    environment = sdk2.parameters.String('Environment',
                                         default=ENV_TESTING,
                                         ui=sdk2.parameters.String.UI('select'),
                                         choices=list((env, env) for env in (ENV_TESTING, ENV_PRODUCTION)),
                                         required=True)
    sandbox_vault_owner = sdk2.parameters.String('Sandbox vault owner',
                                                 default='YANDEX_BUS', required=True)


class BusBaseTask(CommonTaskTrackerMixin, sdk2.Task):

    def fill_task_tracker_config(self):
        if self.parent is not None:
            self.Parameters.tags += [parent_tag for parent_tag in self.parent.Parameters.tags
                                     if parent_tag in INHERITABLE_TAGS]

        enable_solomon = False if SOLOMON_OFF_TAG in self.Parameters.tags else True
        self.task_tracker_config.enable_solomon = enable_solomon
        self.task_tracker_config.solomon_track_start = True
        if enable_solomon:
            self.task_tracker_config.solomon_secret_token = 'sec-01cjvsaf2hrdkfdr97c30dv3t4'
            self.task_tracker_config.solomon_secret_key = 'solomon-token'
            self.task_tracker_config.solomon_secret_version = ''
            self.task_tracker_config.solomon_project = 'bus'
            self.task_tracker_config.solomon_cluster = ENV_TO_CLUSTER_MAPPER[self.Parameters.environment]
            self.task_tracker_config.solomon_service = 'sandbox'
            self.task_tracker_config.solomon_common_labels = {}

        task_name = self.type.__name__
        self.task_tracker_config.source = '.'.join([self._solomon_prefix, task_name])
        self.task_tracker_config.host = task_name


class BusAutoResourceParameters(BusBaseParameters):
    with sdk2.parameters.RadioGroup("Release type of binary",
                                    group="Binary task executor") as binary_executor_release_type:
        binary_executor_release_type.values.stable = binary_executor_release_type.Value(ctt.ReleaseStatus.STABLE,
                                                                                        default=True)
        binary_executor_release_type.values.test = binary_executor_release_type.Value(ctt.ReleaseStatus.TESTING,
                                                                                      default=False)
        binary_executor_release_type.values.none = binary_executor_release_type.Value("Custom", default=False)


class BusBaseTaskAutoResource(BusBaseTask):

    def on_save(self):
        self.Requirements.tasks_resource = None if self.Parameters.binary_executor_release_type == "none" else \
            sdk2.service_resources.SandboxTasksBinary.find(
                attrs={"task_type": self.type.name, "release": self.Parameters.binary_executor_release_type}
            ).first()

    def on_execute(self):
        if self.Parameters.binary_executor_release_type != "none" and self.Requirements.tasks_resource is None:
            raise errors.TaskFailure(
                "Can't find any SandboxTasksBinary resource for task type {} with attribute release=={}".format(
                    self.type.name, self.Parameters.binary_executor_release_type))
