# -*- coding: utf-8 -*-
from __future__ import absolute_import, division, print_function, unicode_literals

import os
from datetime import datetime

from sandbox import sdk2
from sandbox.projects.rasp.bus.BusBaseTask import BusAutoResourceParameters, BusBaseTaskAutoResource


def get_previous_report_date():
    now = datetime.utcnow()

    year = now.year
    month = now.month

    if month == 1:
        year -= 1
        month = 12
    else:
        month -= 1
    return year, month


class BusYearOrderReport(BusBaseTaskAutoResource):

    class Requirements(sdk2.Task.Requirements):
        ram = 1024

    class Parameters(BusAutoResourceParameters):
        year = sdk2.parameters.Integer('Year')
        emails = sdk2.parameters.List('Email', default=[], required=True)

    def on_execute(self):
        os.environ['YENV_TYPE'] = self.Parameters.environment
        os.environ['RASP_VAULT_OAUTH_TOKEN'] = sdk2.Vault.data(self.Parameters.sandbox_vault_owner, 'YAV_OAUTH_TOKEN')
        from travel.rasp.bus.library.locale_setter import RuLocaleHolder
        from travel.rasp.bus.scripts.order_report import year_report

        year = self.Parameters.year

        if year == 0:
            year, _ = get_previous_report_date()

        with RuLocaleHolder():
            year_report(year, self.Parameters.emails)


class BusMonthOrderReport(BusBaseTaskAutoResource):

    class Requirements(sdk2.Task.Requirements):
        ram = 1024

    class Parameters(BusAutoResourceParameters):
        year = sdk2.parameters.Integer('Year')
        month = sdk2.parameters.Integer('Month')
        emails = sdk2.parameters.List('Email', default=[], required=True)

    def on_execute(self):
        os.environ['YENV_TYPE'] = self.Parameters.environment
        os.environ['RASP_VAULT_OAUTH_TOKEN'] = sdk2.Vault.data(self.Parameters.sandbox_vault_owner, 'YAV_OAUTH_TOKEN')
        from travel.rasp.bus.library.locale_setter import RuLocaleHolder
        from travel.rasp.bus.scripts.order_report import month_report

        year = self.Parameters.year
        month = self.Parameters.month

        if year == 0 and month == 0:
            year, month = get_previous_report_date()

        with RuLocaleHolder():
            month_report(year, month, self.Parameters.emails)


class BusDetailedOrderReport(BusBaseTaskAutoResource):

    class Requirements(sdk2.Task.Requirements):
        ram = 1024

    class Parameters(BusAutoResourceParameters):
        year = sdk2.parameters.Integer('Year')
        month = sdk2.parameters.Integer('Month')
        emails = sdk2.parameters.List('Email', default=[], required=True)

    def on_execute(self):
        os.environ['YENV_TYPE'] = self.Parameters.environment
        os.environ['RASP_VAULT_OAUTH_TOKEN'] = sdk2.Vault.data(self.Parameters.sandbox_vault_owner, 'YAV_OAUTH_TOKEN')
        from travel.rasp.bus.library.locale_setter import RuLocaleHolder
        from travel.rasp.bus.scripts.order_report.detailed_report import detailed_report

        year = self.Parameters.year
        month = self.Parameters.month
        if year == 0 and month == 0:
            year, month = get_previous_report_date()

        with RuLocaleHolder():
            detailed_report(year, month, self.Parameters.emails)
