# -*- coding: utf-8 -*-
from __future__ import absolute_import, division, print_function, unicode_literals

import logging

import time
from os import environ

from sandbox import sdk2
from sandbox.common.errors import TaskFailure
from sandbox.common.types import task as ctt

from sandbox.projects.rasp.bus.BusBaseTask import BusBaseTaskAutoResource, BusAutoResourceParameters
from sandbox.projects.rasp.bus.BusResourceUpdate.BusPointkeyFilterUpdate import BusPointkeyFilterUpdate
from sandbox.projects.rasp.bus.BusResourceUpdate.BusTTypeFilterUpdate import BusTTypeFilterUpdate
from sandbox.projects.rasp.bus.BusResourceUpdate.BusAdminDictsUpdate import BusAdminDictsUpdate

RELEASE_STATUS = {
    'testing': ctt.ReleaseStatus.TESTING,
    'production': ctt.ReleaseStatus.STABLE
}


class BusEndpointsUpdate(BusBaseTaskAutoResource):

    class Requirements(sdk2.Task.Requirements):
        disk_space = 1024
        ram = 2048

    class Parameters(BusAutoResourceParameters):
        dry = sdk2.parameters.Bool('Dry', default=True, required=True)
        update_suggests = sdk2.parameters.Bool('Update suggests', default=False, required=True)
        login = sdk2.parameters.String('Login for BusAdmin', default='sandbox', required=True)
        suppliers = sdk2.parameters.List('Suppliers (all if empty)', default=[], required=True)

    def on_execute(self):
        super(BusEndpointsUpdate, self).on_execute()
        environ['YENV_TYPE'] = self.Parameters.environment
        environ['RASP_VAULT_OAUTH_TOKEN'] = sdk2.Vault.data(self.Parameters.sandbox_vault_owner, 'YAV_OAUTH_TOKEN')

        with self.memoize_stage.update_endpoints:
            from travel.rasp.bus.scripts.endpoints_updater import EndpointsUpdater
            updater = EndpointsUpdater()
            all_endpoints = updater.run(login=self.Parameters.login, supplier_codes=self.Parameters.suppliers,
                                        dry=self.Parameters.dry)
            if not all_endpoints and self._available():
                labels = {'description': 'endpoints_load_error'}
                sensors = [self._make_sensor(labels, self._solomon_status_fail, 1, int(time.time()))]
                self._push(sensors)

            if self.Parameters.dry or not self.Parameters.update_suggests:
                return

            tasks = [
                BusPointkeyFilterUpdate(
                    self,
                    environment=self.Parameters.environment,
                    notifications=self.Parameters.notifications,
                    vault_owner=self.Parameters.sandbox_vault_owner,
                ),
                BusTTypeFilterUpdate(
                    self,
                    environment=self.Parameters.environment,
                    notifications=self.Parameters.notifications,
                    vault_owner=self.Parameters.sandbox_vault_owner,
                ),
                BusAdminDictsUpdate(
                    self,
                    environment=self.Parameters.environment,
                    notifications=self.Parameters.notifications,
                    vault_owner=self.Parameters.sandbox_vault_owner,
                )
            ]
            for task in tasks:
                task.enqueue()

            task_ids = [task.id for task in tasks]
            self.Context.wait_for_tasks = task_ids

            raise sdk2.WaitTask(task_ids, ctt.Status.Group.FINISH | ctt.Status.Group.BREAK)

        with self.memoize_stage.update_resources:
            if self.Parameters.dry or not self.Parameters.update_suggests:
                return

            tasks = list(self.find(id=self.Context.wait_for_tasks).limit(len(self.Context.wait_for_tasks)))
            for t in tasks:
                logging.info('create_release for {}'.format(t.id))
                release_id = self.server.release(
                    task_id=t.id,
                    type=RELEASE_STATUS.get(self.Parameters.environment),
                    subject='Releasing {} created on {}'.format(t.id, t.created)
                )

                if release_id is None:
                    raise TaskFailure('Release for task {} failed!'.format(t.id))
                else:
                    logging.info('Released: {}'.format(release_id))

            logging.info('Done!')
