# -*- coding: utf-8 -*-
from __future__ import absolute_import, division, print_function, unicode_literals

import logging
from os import environ
from pathlib2 import Path
from shutil import copy

from sandbox import sdk2
from sandbox.projects.common.ya_deploy.release_integration import ReleaseToNannyAndYaDeployTask2
from sandbox.projects.rasp.bus.BusBaseTask import BusBaseTaskAutoResource, BusAutoResourceParameters
from sandbox.projects.rasp.bus.utils import get_data_resource
from sandbox.projects.rasp.bus.utils.resources import PointkeyFilterResource
from sandbox.projects.Travel.resources import dicts


log = logging.getLogger(__name__)


class BusPointkeyFilterUpdate(ReleaseToNannyAndYaDeployTask2, BusBaseTaskAutoResource):

    RESOURCE_PATH = 'bus_pointkey_filter_resources'
    RESOURCE_BASE_NAME = 'pointkeyfilter.msgpack'
    RESOURCE_TTL = 30

    class Requirements(sdk2.Task.Requirements):
        disk_space = 1024
        ram = 2048

    class Parameters(BusAutoResourceParameters):
        station_resource = sdk2.parameters.Resource(
            'station.bin (last if empty)',
            resource_type=[dicts.TRAVEL_DICT_RASP_STATION_PROD],
            required=False,
        )
        station_to_settlement_resource = sdk2.parameters.Resource(
            'settlement2station.bin (last if empty)',
            resource_type=[dicts.TRAVEL_DICT_RASP_SETTLEMENT_TO_STATION_PROD],
            required=False,
        )

    def on_execute(self):
        super(BusPointkeyFilterUpdate, self).on_execute()
        environ['YENV_TYPE'] = self.Parameters.environment
        environ['RASP_VAULT_OAUTH_TOKEN'] = sdk2.Vault.data(self.Parameters.sandbox_vault_owner, 'YAV_OAUTH_TOKEN')
        from travel.rasp.bus.scripts.pointkey_filter import PointKeyFilterGenerator

        station_resource = self.Parameters.station_resource \
            or get_data_resource(dicts.TRAVEL_DICT_RASP_STATION_PROD)
        station_to_settlement_resource = self.Parameters.station_to_settlement_resource \
            or get_data_resource(dicts.TRAVEL_DICT_RASP_SETTLEMENT_TO_STATION_PROD)
        assert station_resource and station_to_settlement_resource
        station_resource_data = sdk2.ResourceData(station_resource)
        station_to_settlement_resource_data = sdk2.ResourceData(station_to_settlement_resource)
        log.info('Got resource data')

        resource_path = Path(self.RESOURCE_PATH)
        resource_path.mkdir(parents=True, exist_ok=True)

        pointkey_filename = str(resource_path / self.RESOURCE_BASE_NAME)
        pointkey_filter_generator = PointKeyFilterGenerator()
        pointkey_filter_generator.gen(
            station_fn=str(station_resource_data.path.absolute()),
            station_to_settlement_fn=str(station_to_settlement_resource_data.path.absolute()),
            pointkey_fn=pointkey_filename
        )
        log.info('%s generated', self.RESOURCE_BASE_NAME)

        pointkey_filter_resource = PointkeyFilterResource(
            self, description=self.RESOURCE_BASE_NAME, path=self.RESOURCE_BASE_NAME,
            resource_name=self.RESOURCE_BASE_NAME, ttl=self.RESOURCE_TTL
        )
        pk_resource_data = sdk2.ResourceData(pointkey_filter_resource)
        copy(src=pointkey_filename, dst=str(pk_resource_data.path.absolute()))
        pk_resource_data.ready()
        log.info('PointkeyFilterResource %s ready. Done', self.RESOURCE_BASE_NAME)
