import logging
import yaml

from sandbox import sdk2
from sandbox.projects.rasp.utils.email_notifications import EmailNotificationMixin, use_email_notification_params
from sandbox.projects.rasp.utils.juggler import JugglerNotificationMixin, use_juggler_notification_params
from sandbox.sandboxsdk.svn import Arcadia


MAX_DEPTH = 10


class RaspSolomonGenerateAlerts(sdk2.Task, EmailNotificationMixin, JugglerNotificationMixin):
    class Requirements(sdk2.Task.Requirements):
        ram = 2 * 1024
        tasks_resource = sdk2.Task.Requirements.tasks_resource(default=1732381516)

    class Parameters(sdk2.Task.Parameters):
        kill_timeout = 3600

        with sdk2.parameters.Group('General') as general:
            project = sdk2.parameters.String('Solomon project name', required=True, default_value='rasp')
            alert_owner = sdk2.parameters.String('Solomon alert owner name (e.g. robot-rasp)', required=True)
            token_owner = sdk2.parameters.String('Vault token owner name')
            token_name = sdk2.parameters.String('Vault token name', required=True, default_value='SOLOMON_TOKEN')
            config_path = sdk2.parameters.SvnUrl(
                'Path to config in arcadia',
                required=True,
                default_value=Arcadia.trunk_url('travel/rasp/configs/solomon/alerts.yaml')
            )

        _email_notification_params = use_email_notification_params()
        _juggler_notification_params = use_juggler_notification_params()

    def _get_import_files(self, import_config):
        if not import_config:
            return
        prefix = import_config.get('prefix', 'svn+ssh://zomb-sandbox-rw@arcadia.yandex.ru')
        directory = import_config.get('directory', '/arc/trunk/arcadia/')
        for filename in import_config.get('files', []):
            yield prefix + directory + filename

    def _build_config(self, config_path, current_depth=0):
        if current_depth > MAX_DEPTH:
            raise ValueError('Too deep config')
        result_config = yaml.load(Arcadia.cat(config_path))
        for filename in self._get_import_files(result_config.pop('import', None)):
            result_config.update(self._build_config(filename, current_depth + 1))
        return result_config

    def on_save(self):
        super(RaspSolomonGenerateAlerts, self).on_save()
        self.add_email_notifications()
        self.add_juggler_notifications()

    def on_execute(self):
        from travel.rasp.infra.tasks.solomon.alert_updater import SolomonAlertUpdater
        logging.debug('Downloading configs from {}'.format(self.Parameters.config_path))
        token_owner = self.Parameters.token_owner if self.Parameters.token_owner else self.owner
        updater = SolomonAlertUpdater(self.Parameters.project, self._build_config(self.Parameters.config_path),
                                      sdk2.Vault.data(token_owner, self.Parameters.token_name),
                                      self.Parameters.alert_owner)
        updater.run()
