import logging
import requests
import urllib

import sandbox.common.types.task as ctt
from sandbox import sdk2
from sandbox.projects.common.build import YaPackage


SSH_VAULT_NAME = 'RASP_ROBOT_RASP_SSH_KEY'
ADMIN_DEPLOY_SECRET_NAME = 'RASP_ADMIN_DEPLOY_SECRET'
ADMIN_DEPLOY_FLAG_CODE = 20


ADMIN_HOST_BY_RELEASE_STATUS = {
    ctt.ReleaseStatus.UNSTABLE: 'testing.service.admin.rasp.in.yandex-team.ru',
    ctt.ReleaseStatus.TESTING: 'testing.work.admin.rasp.in.yandex-team.ru',
    ctt.ReleaseStatus.PRESTABLE: 'production.service.admin.rasp.in.yandex-team.ru',
    ctt.ReleaseStatus.STABLE: 'production.work.admin.rasp.in.yandex-team.ru'
}


class RaspAdminYaPackage(YaPackage.YaPackage):
    type = 'RASP_ADMIN_YA_PACKAGE'

    @staticmethod
    def split_registry_tag(registry_tag_full):
        if ':' in registry_tag_full:
            registry_url, registry_tag = registry_tag_full.split(':')
        else:
            registry_url, registry_tag = registry_tag_full, 'latest'
        if registry_url.startswith('registry.yandex.net/'):
            registry_url = registry_url[len('registry.yandex.net/'):]

        return registry_url, registry_tag

    def admin_deploy(self, admin_url):
        logging.info('start admin_deploy. url: {admin_url}'.format(admin_url=admin_url))
        try:
            registry_name = self.ctx.get('output_resource_version')
            registry_url, registry_tag = self.split_registry_tag(registry_name)
            registry_full_tag = 'registry.yandex.net/{url}:{tag}'.format(url=registry_url, tag=registry_tag)
            logging.info('registry: {registry}'.format(registry=registry_full_tag))

            response = requests.get("{url}/deploy/?{params}".format(
                url=admin_url,
                params=urllib.urlencode({'registry': registry_full_tag}),
            ))
            logging.info('code:{code} content:{content}'.format(code=response.status_code, content=response.content))
            response.raise_for_status()
        except Exception as ex:
            logging.error('admin_deploy error: {ex}'.format(ex=repr(ex)))

        logging.info('done admin_deploy')

    def on_release(self, additional_parameters):
        try:
            admin_host = ADMIN_HOST_BY_RELEASE_STATUS.get(additional_parameters['release_status'])
            secret = sdk2.ssh.Key(self, self.owner, ADMIN_DEPLOY_SECRET_NAME).key

            response = requests.get('https://{host}/admin/deploy/flags/maintenance/set/?value={code}'.
                                    format(host=admin_host, code=ADMIN_DEPLOY_FLAG_CODE),
                                    verify=False, headers={'ADMIN-DEPLOY-SECRET': secret})
            response.raise_for_status()

            data = response.json()
            if data['status'] == 'ok' and data['value'] == ADMIN_DEPLOY_FLAG_CODE:
                logging.info('set_flag: {}'.format(data['value']))
                YaPackage.YaPackage.on_release(self, additional_parameters)
            else:
                raise Exception('set_flag failed. status: {}. value: {}'.format(data['status'], data['value']))
        except Exception:
            logging.exception('on_release failed')
            raise

    def add_robot_ssh_key(self):
        key = sdk2.ssh.Key(self, self.owner, SSH_VAULT_NAME)
        with open('robot_id_rsa', "w") as f:
            f.write(key.key)

        logging.info('add robot ssh key')

    def add_version(self):
        version = self.ctx.get('checkout_arcadia_from_url').split('rasp_admin/')[1]
        with open('version', "w") as f:
            f.write(version)

        logging.info('add version')

    def on_execute(self):
        self.add_robot_ssh_key()
        self.add_version()
        YaPackage.YaPackage.on_execute(self)
