# coding: utf8
from __future__ import absolute_import, division, print_function, unicode_literals

import logging
from importlib import import_module

from sandbox import sdk2
from sandbox.common.types import task as ctt
from sandbox.projects.common import binary_task

from sandbox.projects.rasp.utils.email_notifications import EmailNotificationMixin, use_email_notification_params


log = logging.getLogger()
log.setLevel(logging.INFO)

RELEASE_STATUS = {
    'testing': ctt.ReleaseStatus.TESTING,
    'production': ctt.ReleaseStatus.STABLE
}


class RaspRunAndReleaseTask(binary_task.LastBinaryTaskRelease, sdk2.Task, EmailNotificationMixin):
    """Задача позволяет запустить подзадачу и зарелизить ее. Используется для релиза ресурсов в Deploy"""
    class Requirements(sdk2.Task.Requirements):
        cores = 1
        ram = 1024

        class Caches(sdk2.Requirements.Caches):  # enable multislot https://wiki.yandex-team.ru/sandbox/clients/#client-tags-multislot
            pass  # means that task do not use any shared caches

    class Parameters(sdk2.Task.Parameters):

        with sdk2.parameters.Group('General') as general:
            task_name = sdk2.parameters.String('Task name', required=True)
            task_path = sdk2.parameters.String('Task path', required=True)
            task_params = sdk2.parameters.JSON('Task params', required=True)
            env_type = sdk2.parameters.String('Env type', required=True, default='testing')
            ext_params = binary_task.binary_release_parameters(stable=True)

            skip_no_resource = sdk2.parameters.Bool('Skip no resources for release', default=False)

        _email_notification_params = use_email_notification_params()

    def on_save(self):
        binary_task.LastBinaryTaskRelease.on_save(self)
        self.add_email_notifications()

    def on_execute(self):
        binary_task.LastBinaryTaskRelease.on_execute(self)

        module = import_module(self.Parameters.task_path)
        task_class = getattr(module, self.Parameters.task_name)

        with self.memoize_stage.run_wait_task:
            logging.info('Run task: {}'.format(self.Parameters.task_name))

            task = task_class(
                self,
                owner=self.Parameters.owner,
                priority=self.Parameters.priority,
                description='Run {} from RASP_RUN_AND_RELEASE_TASK'.format(self.Parameters.task_name),
                **{k: v for k, v in self.Parameters.task_params.items()}
            )

            task.enqueue()
            self.Context.waited_task_id = task.id

            raise sdk2.WaitTask([task.id], ctt.Status.Group.FINISH | ctt.Status.Group.BREAK, wait_all=True)

        logging.info('Release task: {}'.format(self.Parameters.task_name))
        task_id = self.Context.waited_task_id

        release_id = None
        try:
            release_id = self.server.release(
                task_id=task_id,
                type=RELEASE_STATUS.get(self.Parameters.env_type),
                subject='Releasing {}'.format(task_id)
            )
        except Exception:
            log.exception('Release failed!')
            if self.Parameters.skip_no_resource:
                return

        if release_id is None:
            raise Exception('Release for task {} failed!'.format(task_id))
        else:
            logging.info('Released: {}'.format(release_id))
