# coding: utf8
from __future__ import absolute_import, division, print_function, unicode_literals

import logging
import shutil

from pathlib2 import Path

from sandbox import sdk2
from sandbox.common.types.client import Tag
from sandbox.common.types.misc import DnsType
from sandbox.projects.common import binary_task
from sandbox.projects.common.arcadia import sdk as ark_sdk
from sandbox.projects.rasp.utils.email_notifications import EmailNotificationMixin, use_email_notification_params
from sandbox.projects.rasp.resource_types import RaspAdminMarkup
from sandbox.sdk2 import yav

MAX_DEPTH = 10
log = logging.getLogger(__name__)


class RaspInfraUpdateAdminMarkup(binary_task.LastBinaryTaskRelease, sdk2.Task, EmailNotificationMixin):
    ADMIN_CHECKOUT_PATH = Path('admin')
    ARC_SECRET_ID = 'sec-01cjvsaf2hrdkfdr97c30dv3t4'
    ARC_SECRET_KEY = 'arc-token'

    class Requirements(sdk2.Task.Requirements):
        ram = 2 * 1024
        container_resource = 1162079430
        client_tags = Tag.LXC
        privileged = True
        dns = DnsType.DNS64

    class Parameters(sdk2.Task.Parameters):
        kill_timeout = 3600

        ext_params = binary_task.binary_release_parameters(stable=True)

        with sdk2.parameters.Group('General') as general:
            ssh_vault_name = sdk2.parameters.String('Vault item with ssh key for git access', default='RASP_ROBOT_RASP_SSH_KEY')

        _email_notification_params = use_email_notification_params()

    def on_save(self):
        super(RaspInfraUpdateAdminMarkup, self).on_save()
        binary_task.LastBinaryTaskRelease.on_save(self)
        self.add_email_notifications()

    def upload_resource(self, description, path):
        resource = RaspAdminMarkup(self, description=description, path=path, ttl=30, file_type=description)
        resource_data = sdk2.ResourceData(resource)
        shutil.copy(
            src=path,
            dst=resource_data.path.absolute().as_posix(),
        )
        resource_data.ready()

    def on_execute(self):
        binary_task.LastBinaryTaskRelease.on_execute(self)

        from travel.rasp.library.python.shell_utils import cd
        from travel.rasp.infra.tasks.update_admin_markup import build_markup, pack_markup, MARKUP_PATH, BUILD_LOG_PATH

        current_revision = str(ark_sdk.aapi.ArcadiaApi.svn_head())
        path = sdk2.svn.Arcadia.ARCADIA_TRUNK_URL + '/travel/rasp/admin'
        log.info('Try to mount {}'.format(path))

        secret_data = yav.Secret(self.ARC_SECRET_ID).data()
        arc_oauth = secret_data[self.ARC_SECRET_KEY]

        with ark_sdk.mount_arc_path("@".join((path, str(current_revision))),
                                    use_arc_instead_of_aapi=True, arc_oauth_token=arc_oauth) as mount_point:
            shutil.copytree(mount_point, self.ADMIN_CHECKOUT_PATH.absolute().as_posix())

            with cd(self.ADMIN_CHECKOUT_PATH.absolute().as_posix()):
                with sdk2.ssh.Key(self, self.owner, self.Parameters.ssh_vault_name):
                    build_markup(current_revision)

                pack_markup()

                self.upload_resource('admin-markup', MARKUP_PATH)
                self.upload_resource('build-log', BUILD_LOG_PATH)
