# coding: utf8
from __future__ import absolute_import, division, print_function, unicode_literals

import os
import logging
import shutil

from pathlib2 import Path

from sandbox import sdk2
from sandbox.common.types.client import Tag
from sandbox.common.types.misc import DnsType
from sandbox.projects.rasp.utils.email_notifications import EmailNotificationMixin, use_email_notification_params
from sandbox.projects.rasp.resource_types import RaspMordaMarkup
from sandbox.sandboxsdk.svn import Arcadia


MAX_DEPTH = 10
log = logging.getLogger(__name__)


class RaspInfraUpdateMordaMarkup(sdk2.Task, EmailNotificationMixin):
    MORDA_CHECKOUT_PATH = Path('morda')

    class Requirements(sdk2.Task.Requirements):
        ram = 2 * 1024
        tasks_resource = sdk2.Task.Requirements.tasks_resource(default=1372189817)
        container_resource = 1162079430
        client_tags = Tag.LXC
        dns = DnsType.DNS64

    class Parameters(sdk2.Task.Parameters):
        kill_timeout = 3600

        with sdk2.parameters.Group('General') as general:
            aws_url = sdk2.parameters.String('AWS endpoint url', required=True, default_value='https://s3.mds.yandex.net')
            aws_bucket = sdk2.parameters.String('AWS bucket', default_value='rasp')
            aws_prefix = sdk2.parameters.String('AWS prefix', required=True, default_value='s/morda')
            aws_access_key_id_secret_name = sdk2.parameters.String(
                'AWS access_key_id secret name', default='rasp-aws-access-key-id'
            )
            aws_secret_access_key_secret_name = sdk2.parameters.String(
                'AWS secret_access_key secret name', default='rasp-aws-secret-access-key'
            )
            ssh_vault_name = sdk2.parameters.String('Vault item with ssh key for git access', default='RASP_ROBOT_RASP_SSH_KEY')

        _email_notification_params = use_email_notification_params()

    def on_execute(self):
        from travel.rasp.library.python.shell_utils import cd
        from travel.rasp.infra.tasks.update_morda_markup import build_markup, pack_markup, upload_to_s3

        Arcadia.checkout(Arcadia.trunk_url('travel/rasp/morda'), self.MORDA_CHECKOUT_PATH.absolute().as_posix())
        current_revision = Arcadia.get_revision('arcadia:/arc/trunk/arcadia/travel/rasp/morda')
        with cd(self.MORDA_CHECKOUT_PATH.absolute().as_posix()):
            with sdk2.ssh.Key(self, self.owner, self.Parameters.ssh_vault_name):
                build_markup(current_revision)

            pack_markup()

            markup_path = 'markup.tar'
            resource = RaspMordaMarkup(self, description='old-morda-markup', path=markup_path, ttl=30)
            resource_data = sdk2.ResourceData(resource)

            shutil.copy(
                src=markup_path,
                dst=resource_data.path.absolute().as_posix(),
            )
            resource_data.ready()

            os.environ['AWS_ACCESS_KEY_ID'] = sdk2.Vault.data(self.Parameters.aws_access_key_id_secret_name)
            os.environ['AWS_SECRET_ACCESS_KEY'] = sdk2.Vault.data(self.Parameters.aws_secret_access_key_secret_name)
            prefix = '{}/{}'.format(self.Parameters.aws_prefix, current_revision)
            upload_to_s3(self.Parameters.aws_url, self.Parameters.aws_bucket, prefix)

    def on_save(self):
        super(RaspInfraUpdateMordaMarkup, self).on_save()
        self.add_email_notifications()
