# coding: utf8
from __future__ import absolute_import, division, print_function, unicode_literals

import os
import logging
import shutil
import tempfile

from pathlib2 import Path

from sandbox import sdk2
from sandbox.common.types.client import Tag
from sandbox.common.types.misc import DnsType
from sandbox.projects.common import binary_task
from sandbox.projects.common.arcadia import sdk as ark_sdk
from sandbox.projects.rasp.utils.email_notifications import EmailNotificationMixin, use_email_notification_params
from sandbox.projects.rasp.resource_types import RaspTouchMarkup

MAX_DEPTH = 10
log = logging.getLogger(__name__)


class RaspInfraUpdateTouchMarkup(binary_task.LastBinaryTaskRelease, sdk2.Task, EmailNotificationMixin):
    TOUCH_CHECKOUT_PATH = Path('touch')

    class Requirements(sdk2.Task.Requirements):
        ram = 2 * 1024
        container_resource = 1162079430
        client_tags = Tag.LXC
        privileged = True
        dns = DnsType.DNS64

    class Parameters(sdk2.Task.Parameters):
        kill_timeout = 3600

        ext_params = binary_task.binary_release_parameters(stable=True)

        with sdk2.parameters.Group('General') as general:
            aws_url = sdk2.parameters.String('AWS endpoint url', required=True, default_value='https://s3.mds.yandex.net')
            aws_bucket = sdk2.parameters.String('AWS bucket', default_value='rasp')
            aws_prefix = sdk2.parameters.String('AWS prefix', required=True, default_value='s/touch')
            aws_access_key_id_secret_name = sdk2.parameters.String(
                'AWS access_key_id secret name', default='rasp-aws-access-key-id'
            )
            aws_secret_access_key_secret_name = sdk2.parameters.String(
                'AWS secret_access_key secret name', default='rasp-aws-secret-access-key'
            )
            ssh_vault_name = sdk2.parameters.String('Vault item with ssh key for git access', default='RASP_ROBOT_RASP_SSH_KEY')
            update_media = sdk2.parameters.Bool('Update media files', default=False)

        _email_notification_params = use_email_notification_params()

    def on_save(self):
        super(RaspInfraUpdateTouchMarkup, self).on_save()
        binary_task.LastBinaryTaskRelease.on_save(self)
        self.add_email_notifications()

    def on_execute(self):
        binary_task.LastBinaryTaskRelease.on_execute(self)

        from travel.rasp.library.python.shell_utils import cd
        from travel.rasp.infra.tasks.update_touch_markup import (
            build_markup, upload_to_s3, upload_media_to_s3, pack_markup
        )

        current_revision = str(ark_sdk.aapi.ArcadiaApi.svn_head())
        path = sdk2.svn.Arcadia.ARCADIA_TRUNK_URL + '/travel/rasp/touch'
        log.info('Try to mount {}'.format(path))

        # меняем base_cache_dir так как директория по умлочанию read-only в sandbox
        ark_sdk.aapi.ArcadiaApi.base_cache_dir = tempfile.gettempdir()

        with ark_sdk.mount_arc_path("@".join((path, str(current_revision)))) as mount_point:
            shutil.copytree(mount_point, self.TOUCH_CHECKOUT_PATH.absolute().as_posix())

            with cd(self.TOUCH_CHECKOUT_PATH.absolute().as_posix()):
                with sdk2.ssh.Key(self, self.owner, self.Parameters.ssh_vault_name):
                    build_markup(current_revision)

                os.environ['AWS_ACCESS_KEY_ID'] = sdk2.Vault.data(self.Parameters.aws_access_key_id_secret_name)
                os.environ['AWS_SECRET_ACCESS_KEY'] = sdk2.Vault.data(self.Parameters.aws_secret_access_key_secret_name)

                upload_to_s3(self.Parameters.aws_url, self.Parameters.aws_bucket, self.Parameters.aws_prefix, current_revision)

                if self.Parameters.update_media:
                    upload_media_to_s3(self.Parameters.aws_url, self.Parameters.aws_bucket, self.Parameters.aws_prefix)

                pack_markup()

                markup_path = 'markup.tar'
                resource = RaspTouchMarkup(self, description='touch-markup', path=markup_path, ttl=30)
                resource_data = sdk2.ResourceData(resource)

                shutil.copy(
                    src=markup_path,
                    dst=resource_data.path.absolute().as_posix(),
                )
                resource_data.ready()
