import logging
from datetime import date, timedelta

import yaml

from sandbox import sdk2
from sandbox.common.types.task import Status
from sandbox.projects.rasp.utils.email_notifications import EmailNotificationMixin, use_email_notification_params
from sandbox.projects.rasp.load.GenerateAmmoFromYT import RaspLoadGenerateAmmoFromYT
from sandbox.sandboxsdk.svn import Arcadia


class RaspLoadGenerateAmmo(sdk2.Task, EmailNotificationMixin):
    class Parameters(sdk2.Task.Parameters):
        yql_result_save = sdk2.parameters.Bool('Save YQL results to resources?', required=True)
        config_path = sdk2.parameters.SvnUrl('Path to config in arcadia', required=True)
        label_format = sdk2.parameters.String('Label format', default='rasp.{}', required=True)
        yql_token_owner = sdk2.parameters.String('YQL token owner')
        yql_token_name = sdk2.parameters.String('YQL token name', default='YQL_TOKEN')

        _email_notification_params = use_email_notification_params()

    def on_execute(self):
        default_date = str(date.today() - timedelta(days=2))
        logging.debug('Downloading configs from {}'.format(self.Parameters.config_path))
        config = yaml.load(Arcadia.cat(self.Parameters.config_path))
        logging.debug(config)
        subtasks = []
        with self.memoize_stage.generate_ammo:
            for app_name, app_config in config.items():
                try:
                    subtasks.append(
                        RaspLoadGenerateAmmoFromYT(
                            self,
                            owner=self.Parameters.owner,
                            priority=self.Parameters.priority,

                            yql_result_save=self.Parameters.yql_result_save,
                            yql_query=app_config['yql_query'].replace('{date}', default_date),
                            yql_token_owner=self.Parameters.yql_token_owner,
                            yql_token_name=self.Parameters.yql_token_name,

                            ignore_values='\n'.join(app_config.get('ignore_lines', [])),
                            headers=app_config.get('headers', {}),
                            ammo_label=self.Parameters.label_format.format(app_name),

                            enable_email_notifications=False
                        ).enqueue()
                    )
                except Exception:
                    logging.exception('Error while handling {}'.format(app_name))
            logging.debug('Waiting for subtasks')
            raise sdk2.WaitTask(subtasks, Status.Group.FINISH + Status.Group.BREAK, wait_all=True)

    def on_save(self):
        super(RaspLoadGenerateAmmo, self).on_save()
        self.add_email_notifications()
