import logging
import re
from sandbox import sdk2
import sandbox.common.errors as ce
from sandbox.sandboxsdk import environments
from sandbox.projects.tank.load_resources.resources import AMMO_FILE
from sandbox.projects.rasp.utils.ammo_tags import TagBuilder
from sandbox.projects.rasp.utils.email_notifications import EmailNotificationMixin, use_email_notification_params


class RaspLoadGenerateAmmoFromYT(sdk2.Task, EmailNotificationMixin):
    """
        Task for executing YQL-requests and storing results as ammo file
    """

    class Parameters(sdk2.Task.Parameters):
        kill_timeout = 60 * 60 * 3
        with sdk2.parameters.Group('YQL parameters'):
            yql_result_save = sdk2.parameters.Bool('Save YQL results to resources?', default=False)
            yql_query = sdk2.parameters.String('YQL query', multiline=True)
            yql_version = sdk2.parameters.String('YQL syntax version', default='1')
            yql_token_owner = sdk2.parameters.String('YQL token owner')
            yql_token_name = sdk2.parameters.String('YQL token name', default='YQL_TOKEN')
        with sdk2.parameters.Group('Ammo parameters'):
            ignore_values = sdk2.parameters.String(
                'Ignore lines containing:',
                description='Filter patterns, should be separated by newline',
                default='',
                multiline=True
            )
            headers = sdk2.parameters.Dict('Request headers',
                                           description='Headers will be added to every single request',
                                           default={'Connection': 'Close'})
            ammo_label = sdk2.parameters.String(
                'Ammo label',
                description='Ammo will be saved as AMMO_FILE resource with this label'
            )

        _email_notification_params = use_email_notification_params()

    class Requirements(sdk2.Requirements):
        environments = (environments.PipEnvironment('yql', use_wheel=True),
                        environments.PipEnvironment('yandex-yt'))
        disk_space = 1024 * 5

    def filter_line(self, line):
        filter_applied = False
        for item in self.Parameters.ignore_values.split():
            pattern = re.compile(item)
            if pattern.search(line):
                filter_applied = True
                logging.debug('Log line %s was filtered by filter %s', line, item)
        return filter_applied

    def get_yt_logs(self):

        import yql.api.v1.client

        yql_token_owner = self.Parameters.yql_token_owner if self.Parameters.yql_token_owner else self.owner

        yql_token = sdk2.Vault.data(yql_token_owner, self.Parameters.yql_token_name)
        try:
            yql_version = int(self.Parameters.yql_version)
        except ValueError:
            raise ce.TaskError('Please check yql_verion in parameters, it is not integer')

        with yql.api.v1.client.YqlClient(token=yql_token) as yql_client:
            query = yql_client.query(query=self.Parameters.yql_query, syntax_version=yql_version)
            query.run()
            # if not query.results.is_success:
            #     raise ce.TaskError('YQL request failed: ' + '; '.join(str(error) for error in query.results.errors))

            for table in query.get_results():
                table.fetch_full_data()
                for row in table.rows:
                    yield row[0] if not self.filter_line(row[0]) else ''

    def on_execute(self):

        resource_data = sdk2.ResourceData(
            AMMO_FILE(self, 'Resource generated by sandbox task #{}'.format(self.id),
                      'ammo', ammo_label=self.Parameters.ammo_label)
        )
        ammo_file_path = str(resource_data.path)
        headers = '\n'.join(
                ['[{}: {}]'.format(header, value) for header, value in self.Parameters.headers.items()]
            ) if self.Parameters.headers else ''

        tag_builder = TagBuilder()
        with open(ammo_file_path, 'w') as ammo_file:
            if headers:
                ammo_file.write(headers + '\n')
            for line in self.get_yt_logs():
                if line:
                    tag = tag_builder.build_by_yt_log(line)
                    ammo = '{} {}'.format(line, tag)
                    logging.debug('Ammo from YT: %s', ammo)
                    ammo_file.write(ammo + '\n')
        logging.info('File %s is written', ammo_file_path)
        resource_data.ready()

    def on_save(self):
        super(RaspLoadGenerateAmmoFromYT, self).on_save()
        self.add_email_notifications()
