import logging

import yaml

from sandbox import sdk2
from sandbox.common.types.task import Status
from sandbox.projects.rasp.load.StartLoadTesting import RaspLoadStartLoadTesting
from sandbox.projects.rasp.utils.email_notifications import EmailNotificationMixin, use_email_notification_params
from sandbox.sandboxsdk.svn import Arcadia


class RaspLoadStartFromConfig(sdk2.Task, EmailNotificationMixin):
    class Parameters(sdk2.Task.Parameters):
        with sdk2.parameters.Group('General'):
            token_owner = sdk2.parameters.String('Qloud token owner', required=True)
            token_name = sdk2.parameters.String('Qloud token name', required=True)
            config_path = sdk2.parameters.SvnUrl('Path to config in arcadia', required=True)
            run_on_specific_apps = sdk2.parameters.Bool('Run only of specific apps from config', default=False)
            with run_on_specific_apps.value[True]:
                applications_to_run = sdk2.parameters.List(
                    'Application names to run on',
                    description='Tasks will be executed only for these applications. Ex: morda-backend.'
                )

            st_token_name = sdk2.parameters.String(
                'Vault record name with startrek token',
                description='Vault name of record with startrek token',
                required=False
            )
        _email_notification_params = use_email_notification_params()

    def on_execute(self):
        with self.memoize_stage.load_testing_tasks:
            logging.debug('Downloading configs from {}'.format(self.Parameters.config_path))
            config = yaml.load(Arcadia.cat(self.Parameters.config_path))
            logging.debug(config)
            subtasks = []
            for app_name, app_config in config.items():
                if self.Parameters.run_on_specific_apps and app_name not in self.Parameters.applications_to_run:
                    logging.debug('{} not in applications_to_run list, skipping')
                    continue
                try:
                    kwargs = {
                        'token_owner': self.Parameters.token_owner,
                        'token_name': self.Parameters.token_name,
                        'source_qloud_environment': app_config['source'],
                        'components': app_config['components'],
                        'config_source': 'file',
                        'config_content': app_config['tank_config'],
                        'ammo_source': 'label',
                        'ammo_label': app_config['ammo_label'],
                        'only_regression': app_config.get('only_regression', False),
                        'report_type': app_config['report_type'],
                        'additional_message': app_config.get('additional_message', ''),
                    }

                    if 'delete_old' in app_config:
                        kwargs['delete_old'] = app_config['delete_old']
                    if 'acquire_semaphore' in app_config:
                        kwargs['acquire_semaphore'] = app_config['acquire_semaphore']
                    if 'emails' in app_config:
                        kwargs['send_letter'] = True
                        kwargs['mail_recipients'] = app_config['emails']
                    if 'ticket_id' in app_config:
                        kwargs['send_comment'] = True
                        kwargs['ticket_id'] = app_config['ticket_id']
                        kwargs['st_token_name'] = self.Parameters.st_token_name
                    if 'monitoring_config' in app_config:
                        kwargs['use_monitoring'] = True
                        kwargs['monitoring_source'] = 'file'
                        kwargs['monitoring_content'] = app_config['monitoring_config']
                    if 'additional_tankapi_parameters' in app_config:
                        kwargs['config_add_parameters'] = app_config['additional_tankapi_parameters']

                    subtasks.append(
                        RaspLoadStartLoadTesting(
                            self,
                            enable_email_notifications=False,
                            **kwargs
                        ).enqueue()
                    )
                except Exception:
                    logging.exception('Error while handling {}'.format(app_name))
            logging.debug('Waiting for subtasks')
            raise sdk2.WaitTask(subtasks, Status.Group.FINISH + Status.Group.BREAK, wait_all=True)

    def on_save(self):
        super(RaspLoadStartFromConfig, self).on_save()
        self.add_email_notifications()
