import copy
import logging
import time
from datetime import datetime, timedelta

from sandbox import sdk2
from sandbox.common.types.resource import State
from sandbox.common.types.task import Semaphores, Status
from sandbox.projects.rasp.utils.email_notifications import EmailNotificationMixin, use_email_notification_params
from sandbox.projects.tank.LoadTestResults import LoadTestResults
from sandbox.projects.tank.load_resources.resources import AMMO_FILE
from sandbox.projects.tank.ShootViaTankapi import ShootViaTankapi


SEMAPHORE_NAME = 'rasp_load_testing'
QLOUD_DEPLOY_SLEEP_TIMEOUT = 10
DEFAULT_LOCATION = 'SASOVO'
DEFAULT_NETWORK = 'RASP_LOAD_NETS'
TANK_SIZE = '2;1.0;16;1'
TANK_PORT = 8083
TANK_QLOUD_ENVIRONMENT_CONFIG = {
    'componentName': 'tank',
    'componentType': 'standard',
    'properties': {
        'allocationFailThreshold': '0',
        'allocationStrategy': 'dynamic',
        'allowPortoInteraction': 'false',
        'allowedCpuNames': '',
        'componentEnvironment': '',
        'deployPolicy': 'InPlace',
        'diskSerial': '0',
        'diskSize': '20',
        'dnsCache': 'true',
        'dnsNat64': 'false',
        'ediskSerial': '0',
        'ediskSize': '0',
        'failTimeout': '60',
        'fastboneRequired': 'false',
        'generateQloudPeers': 'false',
        'hardwareSegment': 'common',
        'hash': 'sha256:f5f67da8ac37abae858b113c666771f47956ac50f8b73c85d9241747d0ee92a3',
        'healthCheckFall': '5',
        'healthCheckHttpExpectedCode': 'http_2xx',
        'healthCheckHttpUrl': '/',
        'healthCheckInterval': '5000',
        'healthCheckRise': '2',
        'healthCheckTimeout': '2000',
        'healthCheckType': 'http',
        'httpCheckOn': 'true',
        'ioLimit': '0',
        'isolationGroup': 'root',
        'isolationUser': 'root',
        'java': 'false',
        'maxFails': '3',
        'maxInstancesPerHost': '999',
        'minPrepared': '50',
        'network': 'RASP_LOAD_NETS',
        'path': '',
        'preAuthenticate': 'false',
        'profiles': 'production',
        'qloudCoreDumpDirectory': '/coredumps_qloud',
        'qloudCoreDumpFileSizeGb': '0',
        'qloudInitPolicy': 'stable',
        'qloudInitVersion': '591',
        'qloudMaxCoreDumpedStopsRespawnDelay': '0s',
        'qloudMaxCoreDumpsOnDisk': '0',
        'repository': 'registry.yandex.net/load/yandex-tank-pip:1.11.0',
        'size': TANK_SIZE,
        'statusChecksCorrected': 'true',
        'stderr': 'line',
        'stdout': 'line',
        'storage': '',
        'tmpfsDiskSerial': '0',
        'tmpfsDiskSize': '0',
        'unistat': 'false',
        'unistatPath': '/unistat',
        'unistatPort': '80',
        'units': '0',
        'upstreamPort': '80',
        'useDockerUserGroup': 'false',
        'useHealthCheck': 'false',
        'useHttps': 'false',
        'useTorrents': 'false',
        'usedVolumes': ''
    },
    'secrets': [
        {
            'objectId': 'secret.rasp-load-ssh-key',
            'target': '/root/.ssh/id_rsa',
            'used': True
        }
    ],
    'instanceGroups': [
        {
            'location': DEFAULT_LOCATION,
            'units': 1,
            'backup': False,
            'weight': 1
        }
    ],
    'overlays': [],
    'sandboxResources': [],
    'jugglerBundleResources': [],
    'environmentVariables': {},
    'prepareRecipe': {
        'recipe': 'INTERNAL',
        'updateWindow': '100%',
        'doneThreshold': '90%',
        'updateLimit': '100%',
        'updatePeriod': '20s'
    },
    'activateRecipe': {
        'recipe': 'INTERNAL',
        'updateWindow': '100%',
        'doneThreshold': '90%',
        'updateLimit': '20%',
        'updatePeriod': '20s'
    },
    'statusHookChecks': [
        {
            'type': 'http',
            'port': 1,
            'path': '/ping',
            'timeout': 1000
        }
    ],
    'embedJugglerClient': False,
    'componentFeatures': [],
    'upstreamComponents': []
}

DEFAULT_TANK_CONFIG = """phantom:
  load_profile: {load_type: rps, schedule: 'line(1,10,1m) const(10,1m)'}
  address: 'yandex.ru:443'
  ssl: true
  writelog: proto_warning
uploader:
  operator: lunapark
  task: LOAD-318
  job_name:
  job_dsc: 'started from Sandbox'
  meta: {jenkinsbuild: 'http://sandbox.yandex-team.ru/task/{}/view'}
yasm:
  enabled: true
  package: yandextank.plugins.YASM
  panels:
    main:
      host: QLOUD
      tags: itype=qloud;prj={project};tier={component}-1
    tank:
      host: QLOUD
      tags: itype=qloud;prj={project};tier=tank-1
"""

DEFAULT_TANK_INI = """[phantom]
ammofile=
rps_schedule=
uris=
address=
[meta]
operator=lunapark
task=
ver=
job_dsc=started from sandbox with ShootViaTankapi
job_name=
"""

DEFAULT_MONITORING = """<Monitoring>
    <Host address="main-dbm01f.load.rasp.yandex.net" />
    <Host address="main-dbs01f.load.rasp.yandex.net" />
    <Host address="maintenance-dbm01f.load.rasp.yandex.net" />
    <Host address="main-mrs01f.load.rasp.yandex.net" />
    <Host address="main-mrs02f.load.rasp.yandex.net" />
    <Host address="main-mrs03f.load.rasp.yandex.net" />
</Monitoring>
"""


def _shoot_types_radio_group(*args, **kwargs):
    with sdk2.parameters.RadioGroup(*args, **kwargs) as setter:
        setter.values['const'] = setter.Value(value='Constant load shooting', default=True)
        setter.values['line'] = setter.Value(value='Line load shooting with imbalance point')
        setter.values['simple_const_link'] = setter.Value(
            value='Constant load shooting, only http link to lunapark results')
        setter.values['simple_line_link'] = setter.Value(
            value='Line load shooting, only http link to lunapark results')
    return setter


class RaspLoadStartLoadTesting(sdk2.Task, EmailNotificationMixin):
    class Parameters(sdk2.Task.Parameters):
        with sdk2.parameters.Group('General'):
            token_owner = sdk2.parameters.String('Qloud token owner', required=True)
            token_name = sdk2.parameters.String('Qloud token name', required=True)
            source_qloud_environment = sdk2.parameters.String('Source qloud environment name', required=True)
            components = sdk2.parameters.List('Components to be forked', required=True, default=['main'])
            load_qloud_environment = sdk2.parameters.String(
                'Target qloud environment name (default (if empty) - "load" evnironment near source)'
            )
            target_component = sdk2.parameters.String('Target component name', default='main')
            target_host = sdk2.parameters.String(
                'Target host (default (if empty) - first FQDN of target component hosts from load enviroment)'
            )
            dont_touch_load_environment = sdk2.parameters.Bool(
                'Don\'t touch (free or recreate) load environment if it already exists?', default=False
            )
            delete_qloud_load_environment = sdk2.parameters.Bool(
                'Delete temporary load environment after completing tasks', default=False
            )
            acquire_semaphore = sdk2.parameters.Bool(
                'Acquire semaphore to run one load testing per time', default=True
            )

        with sdk2.parameters.Group('Tank config parameters') as tank_config_block:
            with sdk2.parameters.RadioGroup('Tank config source', required=True) as config_source:
                config_source.values['file'] = config_source.Value(value='Upload config content')
                config_source.values['arcadia'] = config_source.Value(value='Config from Arcadia')
                with config_source.value['file']:
                    config_content = sdk2.parameters.String('Tank config contents', multiline=True,
                                                            default_value=DEFAULT_TANK_CONFIG)
                with config_source.value['arcadia']:
                    config_arc_path = sdk2.parameters.String(
                        "Path to tank config relative to the root",
                        description='for example sandbox/projects/tank/szypulka_test_shoot/configs/load_const.yaml')
                config_add_parameters = sdk2.parameters.String(
                    'Additional tankapi parameters',
                    description='Will be added to cmd, override options set in tank config. '
                                'Format: -o phantom.writelog=all -o autostop.autostop=http(5xx,10%,1m)')

        with sdk2.parameters.Group('Monitoring config parameters') as monitoring_block:
            use_monitoring = sdk2.parameters.Bool('Collect host metrics other then default?',
                                                  description='Add config for metrics collection during the shoot')
            with use_monitoring.value[True]:
                with sdk2.parameters.RadioGroup('Monitoring config source', required=True) as monitoring_source:
                    monitoring_source.values['in_config'] = monitoring_source.Value(value='Already set in tank config',
                                                                                    default=True)
                    monitoring_source.values['file'] = monitoring_source.Value(value='Upload config content')
                    monitoring_source.values['arcadia'] = monitoring_source.Value(value='Config from Arcadia')
                with monitoring_source.value['file']:
                    monitoring_content = sdk2.parameters.String('Monitoring config contents', multiline=True,
                                                                default_value=DEFAULT_MONITORING)
                with monitoring_source.value['arcadia']:
                    monitoring_arc_path = sdk2.parameters.String(
                        'Path to monitoring config relative to the root',
                        description='for example  sandbox/projects/tank/szypulka_test_shoot/configs/monitoring.xml')

        with sdk2.parameters.Group('Ammo parameters') as ammo_block:
            with sdk2.parameters.RadioGroup('Ammo source', required=True) as ammo_source:
                ammo_source.values['in_config'] = ammo_source.Value(value='Already set in tank config',
                                                                    default=True)
                ammo_source.values['resource'] = ammo_source.Value(value='Sandbox resource')
                ammo_source.values['arcadia'] = ammo_source.Value(value='Ammo from Arcadia')
                ammo_source.values['label'] = ammo_source.Value(value='Get resource automatically by label')
            with ammo_source.value['arcadia']:
                ammo_arc_path = sdk2.parameters.String('Path to ammo')
            with ammo_source.value['resource']:
                ammo_resource = sdk2.parameters.Resource('Resource with ammo', state=State.READY)
            with ammo_source.value['label']:
                ammo_label = sdk2.parameters.String('Label for resource')

        with sdk2.parameters.Group('Load testing report') as report_block:
            only_regression = sdk2.parameters.Bool(
                'Report only failed KPIs in regression?',
                description='Reports will be sent only if regression KPI failed',
                default=False
            )
            report_type = _shoot_types_radio_group(
                'Type of report',
                description='Report content depends on shoot type',
                required=True
            )
            additional_message = sdk2.parameters.String(
                'Message that will be added to the report',
                multiline=True
            )
            send_comment = sdk2.parameters.Bool(
                'Send comment to Startrek',
                description='Send comment with test report to Startrek ticket',
                default=False
            )
            with send_comment.value[True]:
                ticket_id = sdk2.parameters.String(
                    'ST ticket',
                    description='Startrek ticket where comment will be added, i.e. LOAD-318',
                    required=False
                )
                st_token_name = sdk2.parameters.String(
                    'Vault record name with startrek token',
                    description='Vault name of record with startrek token',
                    default='lunapark-startrek-token',
                    required=False
                )

            send_letter = sdk2.parameters.Bool(
                'Send e-mail',
                description='Send letter with test report',
                default=False
            )
            with send_letter.value[True]:
                mail_recipients = sdk2.parameters.List(
                    'Mail to',
                    sdk2.parameters.String('email', description='Emails where report will be sent to'),
                    required=False
                )

        _email_notification_params = use_email_notification_params()

    @staticmethod
    def setup_environment(environment, destination_name, required_components):
        environment['comment'] = 'Create load environment via sandbox-task'
        for component in environment.get('components', []):
            name = component.get('componentName')
            if name == 'tank':
                raise ValueError('Component with name "tank" already exists')
            component['instanceGroups'] = [
                {
                    'location': DEFAULT_LOCATION,
                    'units': 1 if name in required_components else 0,
                    'backup': False,
                    'weight': 1
                }
            ]
            component['properties']['network'] = DEFAULT_NETWORK
            component['environmentVariables']['YENV_TYPE'] = 'stress'
            component_environment = component['properties'].get('componentEnvironment', '').splitlines()
            for index, value in enumerate(component_environment):
                if value.startswith('YENV_TYPE='):
                    component_environment[index] = 'YENV_TYPE=stress'
                    break
            else:
                component_environment.append('YENV_TYPE=stress')
            component['properties']['componentEnvironment'] = '\n'.join(component_environment) + '\n'
        environment['objectId'] = destination_name
        environment['components'].append(copy.deepcopy(TANK_QLOUD_ENVIRONMENT_CONFIG))

    def _create_qloud_load_environment(self, qloud_api, source_env, dest_name, components):
        if qloud_api.environment_exists(dest_name):
            logging.info('Destination environment {} already exists'.format(dest_name))
            if self.Parameters.dont_touch_load_environment:
                return
        else:
            app_name, env_name = dest_name.rsplit('.', 1)
            qloud_api.create_environment(app_name, env_name)
        env_dump = qloud_api.dump_environment(source_env)
        logging.debug('Source environment dump: %r', env_dump)
        self.setup_environment(env_dump, dest_name, components)
        logging.debug('Destination environment dump: %r', env_dump)
        qloud_api.upload_environment(env_dump)

    @staticmethod
    def _wait_for_environment_deployment(qloud_api, environment_name, timeout=timedelta(minutes=20)):
        start = datetime.now()
        while datetime.now() - start < timeout:
            time.sleep(QLOUD_DEPLOY_SLEEP_TIMEOUT)
            if qloud_api.get_environment_info(environment_name)['status'] == 'DEPLOYED':
                return
        raise ValueError('Qloud environment is not deployed in a given time')

    @staticmethod
    def _get_fqdns(qloud_api, environment_name, component_name, port=None):
        instances = qloud_api.get_environment_info(environment_name)['components'][component_name]['runningInstances']
        if port is None:
            return [instance['host'] for instance in instances]
        return ['{}:{}'.format(instance['host'], port) for instance in instances]

    @staticmethod
    def _free_qloud_environment(qloud_api, environment_name):
        env_dump = qloud_api.dump_environment(environment_name)
        env_dump['comment'] = 'Free resources'
        for component in env_dump.get('components', []):
            for instance_group in component['instanceGroups']:
                instance_group['units'] = 0
        qloud_api.upload_environment(env_dump)

    def _run_subtasks(self, tanks, target_host, project, main_component):
        with self.memoize_stage.shoot_via_tankapi:
            ammo_source = self.Parameters.ammo_source
            ammo_resource = self.Parameters.ammo_resource
            if self.Parameters.ammo_source == 'label':
                ammo_resource = AMMO_FILE.find(
                    state=State.READY,
                    attrs={'ammo_label': self.Parameters.ammo_label}
                ).order(-AMMO_FILE.id).first().id
                ammo_source = 'resource'
            config_content = self.Parameters.config_content.replace('{target}', target_host)\
                                                           .replace('{project}', project)\
                                                           .replace('{component}', main_component)
            load_test_task = ShootViaTankapi(
                self,

                use_public_tanks=False,
                tanks=tanks,

                config_source=self.Parameters.config_source,
                config_content=config_content,
                config_arc_path=self.Parameters.config_arc_path,
                config_add_parameters=self.Parameters.config_add_parameters,

                use_monitoring=self.Parameters.use_monitoring,
                monitoring_content=self.Parameters.monitoring_content,
                monitoring_arc_path=self.Parameters.monitoring_arc_path,

                ammo_source=ammo_source,
                ammo_arc_path=self.Parameters.ammo_arc_path,
                ammo_resource=ammo_resource,

            ).enqueue()
            raise sdk2.WaitTask([load_test_task], Status.Group.FINISH + Status.Group.BREAK)
        load_test_task = self.find(ShootViaTankapi).first()

        with self.memoize_stage.get_load_testing_results:
            load_test_result_task = LoadTestResults(
                self,
                only_regression=self.Parameters.only_regression,
                shoot_id=load_test_task.Parameters.lunapark_job_id,
                report_type=self.Parameters.report_type,
                additional_message=self.Parameters.additional_message,
                send_comment=self.Parameters.send_comment,
                ticket_id=self.Parameters.ticket_id,
                st_token_name=self.Parameters.st_token_name,
                send_letter=self.Parameters.send_letter,
                mail_recipients=self.Parameters.mail_recipients,
            ).enqueue()
            raise sdk2.WaitTask([load_test_result_task], Status.Group.FINISH + Status.Group.BREAK)

    def on_enqueue(self):
        if self.Parameters.acquire_semaphore:
            logging.info('Acquiring semaphore')
            self.Requirements.semaphores = Semaphores(
                acquires=[Semaphores.Acquire(name=SEMAPHORE_NAME, capacity=1)],
            )

    def _validate_load_qloud_environment(self):
        if self.Parameters.load_qloud_environment:
            value = self.Parameters.load_qloud_environment
            if value.endswith('production'):
                raise ValueError('Load enviroment can\'t be "production"')
            if value.endswith('testing') and not self.Parameters.dont_touch_load_environment:
                raise ValueError('Testing enviroment can\'t be replaced by source environment')

    def _get_load_environment_name(self):
        self._validate_load_qloud_environment()
        if self.Parameters.load_qloud_environment:
            return self.Parameters.load_qloud_environment
        else:
            return self.Parameters.source_qloud_environment.rsplit('.', 1)[0] + '.load'

    def on_execute(self):
        from projects.rasp.qloud.api import QloudPublicApi
        qloud_api = QloudPublicApi(token=sdk2.Vault.data(
            self.Parameters.token_owner,
            self.Parameters.token_name
        ))
        load_environment_name = self._get_load_environment_name()
        logging.info('Load qloud environment: {}'.format(load_environment_name))
        with self.memoize_stage.create_qloud_environment:
            logging.info('Creating temporary qloud environment: {}'.format(load_environment_name))
            self._create_qloud_load_environment(
                qloud_api,
                self.Parameters.source_qloud_environment,
                load_environment_name,
                self.Parameters.components,
            )
            logging.info('Waiting for environment deployment')
            self._wait_for_environment_deployment(qloud_api, load_environment_name)
            logging.info('{} was successfully deployed'.format(load_environment_name))

        tank_instances = self._get_fqdns(qloud_api, load_environment_name, 'tank', port=TANK_PORT)
        logging.info('Tank instance hosts are: {}'.format(tank_instances))

        if self.Parameters.target_host:
            target_host = self.Parameters.target_host
        else:
            target_host = self._get_fqdns(qloud_api, load_environment_name, self.Parameters.target_component)[0]
        logging.info('Target host: {}'.format(target_host))

        logging.info('Running load subtasks')
        self._run_subtasks(
            tank_instances,
            target_host,
            project=load_environment_name,
            main_component=self.Parameters.target_component
        )

        if self.Parameters.delete_qloud_load_environment:
            with self.memoize_stage.delete_qloud_environment:
                logging.info('Deleting qloud temporary environment: {}'.format(load_environment_name))
                qloud_api.delete_environment(load_environment_name)
        else:
            with self.memoize_stage.free_qloud_load_environment:
                if not self.Parameters.dont_touch_load_environment:
                    logging.info('Freeing qloud temporary environment: {}'.format(load_environment_name))
                    self._free_qloud_environment(qloud_api, load_environment_name)

    def on_save(self):
        super(RaspLoadStartLoadTesting, self).on_save()
        self.add_email_notifications()
