# coding: utf8
from __future__ import absolute_import, division, print_function, unicode_literals

import logging
import os

from sandbox import sdk2
from sandbox.projects.common import binary_task
from sandbox.projects.rasp.utils.juggler import JugglerNotificationMixin, use_juggler_notification_params

log = logging.getLogger(__name__)
log.setLevel(logging.INFO)


class TravelContentAdminUpdateDicts(binary_task.LastBinaryTaskRelease, JugglerNotificationMixin, sdk2.Task):
    class Requirements(sdk2.Task.Requirements):
        cores = 1
        ram = 1 * 1024

        class Caches(sdk2.Requirements.Caches):  # enable multislot https://wiki.yandex-team.ru/sandbox/clients/#client-tags-multislot
            pass  # means that task do not use any shared caches

    class Parameters(sdk2.Parameters):
        kill_timeout = 3600

        ext_params = binary_task.binary_release_parameters(stable=True)

        vault_oauth = sdk2.parameters.String('Vault oauth name', required=True, default='rasp-robot-vault-oauth')
        env_type = sdk2.parameters.String('Env type', required=True, default='production')

        _juggler_notification_params = use_juggler_notification_params()

    def on_save(self):
        binary_task.LastBinaryTaskRelease.on_save(self)
        self.add_juggler_notifications(environment=self.Parameters.env_type)

    def on_execute(self):
        vault_oauth = sdk2.Vault.data(self.Parameters.vault_oauth)
        os.environ['RASP_VAULT_OAUTH_TOKEN'] = vault_oauth
        os.environ['YENV_TYPE'] = self.Parameters.env_type
        os.environ['DJANGO_SETTINGS_MODULE'] = 'travel.marketing.content.admin.scripts.local_settings'

        from travel.marketing.content.admin.scripts.load_project import load_project
        load_project()

        from travel.marketing.content.admin.scripts.update_dicts import update

        settlements = sdk2.Resource['TRAVEL_DICT_RASP_SETTLEMENT_PROD'].find().first()
        countries = sdk2.Resource['TRAVEL_DICT_RASP_COUNTRY_PROD'].find().first()
        regions = sdk2.Resource['TRAVEL_DICT_RASP_REGION_PROD'].find().first()

        log.info('Resources: {} {}'.format(settlements, countries))

        update(
            countries_path=sdk2.ResourceData(countries).path,
            settlements_path=sdk2.ResourceData(settlements).path,
            regions_path=sdk2.ResourceData(regions).path
        )

        log.info('DONE')
