# coding: utf8
from __future__ import absolute_import, division, print_function, unicode_literals

import logging

from requests.exceptions import HTTPError

from sandbox import sandboxsdk
from sandbox import sdk2
from sandbox.common.types import task as ctt
from sandbox.projects.rasp.utils.email_notifications import EmailNotificationMixin, use_email_notification_params
from sandbox.projects.rasp.utils.retry_task import retry_task_on


class RaspQloudRestartEnvironment(sdk2.Task, EmailNotificationMixin):
    WILDCARD = '*'

    class Requirements(sdk2.Task.Requirements):
        environments = (
            sandboxsdk.environments.PipEnvironment('requests'),
        )

    class Parameters(sdk2.Task.Parameters):
        kill_timeout = 15 * 60

        qloud_component_path = sdk2.parameters.String(
            'Qloud component path (for ex: rasp.blablacar.testing.foo)',
            required=True,
        )
        comment = sdk2.parameters.String('Comment', required=False, default='', multiline=True)
        registry_tag = sdk2.parameters.String('Registry Url', required=True)

        token_name = sdk2.parameters.String('Token Vault Name', required=True)
        token_owner = sdk2.parameters.String('Token Vault Owner', required=True)

        _email_notification_params = use_email_notification_params()

    def on_enqueue(self):
        environment = '.'.join(self.Parameters.qloud_component_path.split('.')[:-1])
        name = 'qloud_update_{}'.format(environment)
        self.Requirements.semaphores = ctt.Semaphores(
            acquires=[ctt.Semaphores.Acquire(name=name, capacity=1)]
        )

    @staticmethod
    def split_registry_tag(registry_tag_full):
        if ':' in registry_tag_full:
            registry_url, registry_tag = registry_tag_full.split(':')
        else:
            registry_url, registry_tag = registry_tag_full, 'latest'
        if registry_url.startswith('registry.yandex.net/'):
            registry_url = registry_url[len('registry.yandex.net/'):]

        return registry_url, registry_tag

    @retry_task_on(HTTPError)
    def on_execute(self):
        from projects.rasp.qloud.api import QloudPublicApi, QloudPrivateApi, ComponentSpecification
        token = sdk2.Vault.data(self.Parameters.token_owner, self.Parameters.token_name)
        qloud_api = QloudPublicApi(token)
        component_spec = ComponentSpecification.parse_from_path(self.Parameters.qloud_component_path)

        env_dump = qloud_api.dump_environment(component_spec.environment_id)

        registry_url, registry_tag = self.split_registry_tag(self.Parameters.registry_tag)
        registry_full_tag = 'registry.yandex.net/{url}:{tag}'.format(url=registry_url, tag=registry_tag)
        docker_hash = QloudPrivateApi(token).docker_hash(registry_url, registry_tag)

        logging.info('Updating component with path %s, registry %s and tag %s', self.Parameters.qloud_component_path,
                     registry_url, registry_tag)

        for component in self._iter_components_to_update(env_dump, component_spec.name):
            component['properties']['hash'] = docker_hash
            component['properties']['repository'] = registry_full_tag

        env_dump['comment'] = self.Parameters.comment or ''
        qloud_api.upload_environment(env_dump)
        logging.info('Done')

    def _iter_components_to_update(self, env, component_name):
        component_found = False
        for component in env.get('components', []):
            if (component['componentName'] == component_name) or component_name == self.WILDCARD:
                component_found = True
                yield component

        if not component_found:
            msg = 'No component for path {} were found'.format(self.Parameters.qloud_component_path)
            logging.error(msg)
            raise Exception(msg)

    def on_save(self):
        super(RaspQloudRestartEnvironment, self).on_save()
        self.add_email_notifications()
