import logging

from sandbox import sandboxsdk
from sandbox import sdk2
from sandbox.common.types import task as ctt
from sandbox.projects.rasp.utils.email_notifications import EmailNotificationMixin, use_email_notification_params


class RaspQloudUpdateResources(sdk2.Task, EmailNotificationMixin):
    class Requirements(sdk2.Task.Requirements):
        environments = (
            sandboxsdk.environments.PipEnvironment('requests'),
        )

    class Parameters(sdk2.Task.Parameters):
        kill_timeout = 3600

        qloud_environment = sdk2.parameters.String('Qloud environment name', required=True)
        resources = sdk2.parameters.Resource('Resources to be updated', multiple=True)
        variables = sdk2.parameters.Dict('Variables to be updated')

        token_owner = sdk2.parameters.String('Token owner name name', required=True)
        token_name = sdk2.parameters.String('Token name', required=True)

        _email_notification_params = use_email_notification_params()

    @staticmethod
    def _set_qloud_component_resource(resource_dump, target_resource_ids):
        resource = sdk2.Resource.find(id=resource_dump['id']).first()
        try:
            target_resource_id = target_resource_ids[(resource.type, resource.path)]
        except KeyError:
            return False
        else:
            resource_dump['id'] = target_resource_id
            return True

    def on_enqueue(self):
        name = 'qloud_update_{}'.format(self.Parameters.qloud_environment)
        self.Requirements.semaphores = ctt.Semaphores(
            acquires=[ctt.Semaphores.Acquire(name=name, capacity=1)]
        )

    def on_execute(self):
        from projects.rasp.qloud.api import QloudPublicApi
        qloud_api = QloudPublicApi(token=sdk2.Vault.data(
            self.Parameters.token_owner,
            self.Parameters.token_name
        ))
        target_resource_ids = {(resource.type, resource.path): int(resource) for resource in self.Parameters.resources}
        env_dump = qloud_api.dump_environment(self.Parameters.qloud_environment)
        logging.debug('Env dump before: %r', env_dump)
        updated_resources = tuple(
            resource_dump
            for component_dump in env_dump.get('components', [])
            for resource_dump in component_dump.get('sandboxResources', [])
            if self._set_qloud_component_resource(resource_dump, target_resource_ids)
        )
        for component_dump in env_dump.get('components', []):
            environment_variables = component_dump.get('environmentVariables', None)
            if not environment_variables:
                continue
            for var_name, var_value in self.Parameters.variables.items():
                environment_variables[var_name] = var_value

        logging.debug('Env dump after: %r', env_dump)
        if updated_resources:
            qloud_api.upload_environment(env_dump)
        else:
            logging.error("Env doesn't have resource requrements")

    def on_save(self):
        super(RaspQloudUpdateResources, self).on_save()
        self.add_email_notifications()
