# coding: utf8
from __future__ import unicode_literals, absolute_import, division, print_function

import logging
from collections import namedtuple

import requests
from requests.adapters import HTTPAdapter
from requests.packages.urllib3.util.retry import Retry


LOGGER = logging.getLogger(__name__)
DEFAULT_TIMEOUT = 20


class ComponentSpecification(namedtuple('ComponentSpecification', ['project', 'application', 'environment', 'name'])):
    @classmethod
    def parse_from_path(cls, path):
        return cls(*path.split('.'))

    @property
    def environment_id(self):
        return '{self.project}.{self.application}.{self.environment}'.format(self=self)

    def with_environment(self, environment):
        return self._replace(environment=environment)


class ApiBase(object):
    def __init__(self, token, timeout=DEFAULT_TIMEOUT):
        self._session = requests.Session()
        self._session.headers.update({'Authorization': 'OAuth {}'.format(token)})
        retries = Retry(total=3, backoff_factor=0.5)
        adapter = HTTPAdapter(max_retries=retries)
        self._session.mount('http://', adapter)
        self._session.mount('https://', adapter)
        self.timeout = timeout

    def _make_request(self, method, *args, **kwargs):
        kwargs.setdefault('timeout', self.timeout)
        LOGGER.info('Make request: method=[%s], args=[%r], kwargs=[%r]', method, args, kwargs)
        return getattr(self._session, method)(*args, **kwargs)

    def _process_text_query(self, method, *args, **kwargs):
        response = self._make_request(method, *args, **kwargs)
        try:
            response.raise_for_status()
        except requests.HTTPError:
            LOGGER.exception('Api Error')
            LOGGER.error('Response body: %s', response.text)
            raise

        return response.text

    def _process_json_query(self, method, *args, **kwargs):
        response = self._make_request(method, *args, **kwargs)
        try:
            response.raise_for_status()
        except requests.HTTPError:
            LOGGER.exception('Api Error')
            LOGGER.error('Response body: %s', response.text)
            raise

        return response.json()


class QloudPublicApi(ApiBase):
    URL = 'https://platform.yandex-team.ru/api/v1/'

    def dump_environment(self, environment_id):
        url = self.URL + 'environment/dump/{}'.format(environment_id)
        return self._process_json_query('get', url)

    def upload_environment(self, environment_dump, verbose=False):
        path = 'environment/upload/return-header' if verbose else 'environment/upload'
        return self._process_json_query('post', self.URL + path, json=environment_dump)

    def create_environment(self, application_id, environment_name, engine='platform'):
        url = self.URL + 'environment/new'
        return self._process_json_query('post', url, data={
            'applicationId': application_id,
            'environmentName': environment_name,
            'engine': engine,
        })

    def get_environment_info(self, environment_id):
        url = self.URL + 'environment/stable/{}'.format(environment_id)
        return self._process_json_query('get', url)

    def get_environment_version_info(self, environment_id, version):
        url = self.URL + 'environment/stable/{}/{}'.format(environment_id, version)
        return self._process_json_query('get', url)

    def delete_environment(self, environment_id):
        url = self.URL + 'environment/stable/{}'.format(environment_id)
        self._process_text_query('delete', url)

    def environment_exists(self, environment_id):
        try:
            url = self.URL + 'environment/dump/{}'.format(environment_id)
            response = self._make_request('get', url)
            if response.status_code == 404:
                return False
            response.raise_for_status()
        except requests.HTTPError:
            LOGGER.exception('Api Error')
            LOGGER.error('Response body: %s', response.text)
            raise
        else:
            return True

    def get_component_info(self, component_id):
        environment_id, component_names_str = component_id.rsplit('.', 1)
        component_names = {name for name in component_names_str.split(',') if name}
        dump = self.get_environment_info(environment_id)

        for current_component_name, component in dump.get('components', {}).items():
            if (current_component_name in component_names) or ('*' in component_names):
                return component
        LOGGER.debug('%s not found in %s! Environment info:\n%s', component_names_str, environment_id, dump)
        return None


class QloudPrivateApi(ApiBase):
    PRIVATE_API_URL = 'https://platform.yandex-team.ru/api/'

    def docker_hash(self, registry, tag='latest'):
        """
        Докер hash не тоже самое что и digest, поэтому тут у нас отдельная ручка
        """
        return self._process_text_query(
            'get', self.PRIVATE_API_URL + 'docker/hash',
            params={
                'registryUrl': 'registry.yandex.net/{}'.format(registry),
                'tag': tag,
            }
        )


class RegistryApi(ApiBase):
    URL = 'https://registry.yandex.net/v2/'

    def get_list_tags(self, repo):
        url = self.URL + repo + '/tags/list'

        return self._process_json_query('get', url)

    def get_digest_and_manifests(self, repo, referece):
        """
        :param referece: tag or digest
        """
        url = self.URL + repo + '/manifests/' + referece
        response = self._make_request('get', url)
        try:
            response.raise_for_status()
        except requests.HTTPError:
            LOGGER.exception('Api Error')
            LOGGER.error('Response body: %s', response.text)
            raise

        manifests = response.json()
        return response.headers['Docker-Content-Digest'], manifests
