# coding: utf8
from __future__ import absolute_import, division, print_function, unicode_literals

import logging

from sandbox import sdk2
from sandbox.common.types.misc import DnsType
from sandbox.projects.common import binary_task
from sandbox.projects.rasp.utils.juggler import JugglerNotificationMixin, use_juggler_notification_params
from sandbox.projects.rasp.utils.email_notifications import EmailNotificationMixin, use_email_notification_params
from sandbox.projects.rasp.utils.rasp_scripts import RaspScriptsTask, use_rasp_scripts_params


log = logging.getLogger(__name__)
log.setLevel(logging.INFO)


class RaspBuildKrasnodarBusSchedule(
    binary_task.LastBinaryTaskRelease,
    sdk2.Task,
    EmailNotificationMixin,
    JugglerNotificationMixin,
    RaspScriptsTask
):
    class Requirements(sdk2.Task.Requirements):
        cores = 16
        ram = 4 * 1024
        dns = DnsType.DNS64

        class Caches(sdk2.Requirements.Caches):  # enable multislot https://wiki.yandex-team.ru/sandbox/clients/#client-tags-multislot
            pass  # means that task do not use any shared caches

    class Parameters(sdk2.Task.Parameters):
        kill_timeout = 3600 * 3

        ext_params = binary_task.binary_release_parameters(stable=True)

        with sdk2.parameters.Group('General') as general:
            # Это 83.239.56.140 переведённый в IPv6
            # https://wiki.yandex-team.ru/Travel/Rasp/Dev/Network/#nat64/dns64
            host = sdk2.parameters.String(
                'Host of krasnodar bus api',
                required=False,
                default_value='http://[64:ff9b::53ef:388c]:8080/'
            )
            pool_size = sdk2.parameters.Integer('Size of pool for multiprocessing', required=False, default_value=4)
            mds_path = sdk2.parameters.String('Path to output file in mds', required=True, default_value='schedule/bus/krasnodar.xml')
            days_count = sdk2.parameters.Integer('Count of schedule days', required=True, default_value=2)
            days_shift_from = sdk2.parameters.Integer('Start of load days range', required=True, default_value=1)
            days_shift_to = sdk2.parameters.Integer('End of load days range', required=True, default_value=2)

        _rasp_scripts_params = use_rasp_scripts_params()
        _email_notification_params = use_email_notification_params()
        _juggler_notification_params = use_juggler_notification_params()

    def on_save(self):
        binary_task.LastBinaryTaskRelease.on_save(self)
        self.add_email_notifications()
        self.add_juggler_notifications(environment=self.Parameters.env_type)

    def on_execute(self):
        binary_task.LastBinaryTaskRelease.on_execute(self)

        self.set_common_environ()

        from travel.rasp.rasp_scripts.scripts.schedule.krasnodar import run

        run(
            days_count=self.Parameters.days_count,
            days_shift_from=self.Parameters.days_shift_from,
            days_shift_to=self.Parameters.days_shift_to,
            pool_size=self.Parameters.pool_size,
            host=self.Parameters.host,
            mds_path=self.Parameters.mds_path
        )

        log.info('DONE')
