# coding: utf8
from __future__ import absolute_import, division, print_function, unicode_literals

import logging

from sandbox import sdk2
from sandbox.common.types.misc import DnsType
from sandbox.projects.common import binary_task
from sandbox.projects.rasp.utils.email_notifications import EmailNotificationMixin, use_email_notification_params
from sandbox.projects.rasp.utils.juggler import JugglerNotificationMixin, use_juggler_notification_params
from sandbox.projects.rasp.utils.rasp_scripts import RaspScriptsTask, use_rasp_scripts_params

log = logging.getLogger(__name__)
log.setLevel(logging.INFO)


class RaspBuildOdnakassaSchedule(
    binary_task.LastBinaryTaskRelease,
    sdk2.Task,
    EmailNotificationMixin,
    JugglerNotificationMixin,
    RaspScriptsTask
):
    class Requirements(sdk2.Task.Requirements):
        cores = 1
        ram = 1 * 1024
        dns = DnsType.DNS64

        class Caches(sdk2.Requirements.Caches):  # enable multislot https://wiki.yandex-team.ru/sandbox/clients/#client-tags-multislot
            pass  # means that task do not use any shared caches

    class Parameters(sdk2.Task.Parameters):
        kill_timeout = 3600

        ext_params = binary_task.binary_release_parameters(stable=True)

        with sdk2.parameters.Group('General') as general:
            base_url = sdk2.parameters.String('Odnakassa base url', required=False,
                                              default='https://api-gds.odnakassa.ru/ride/list')

            cities = sdk2.parameters.String('City codes (comma separated)', required=False, default='1375,1193,1227')
            start_date = sdk2.parameters.String('Schedule start date in ISO 8601 format. (default=today)',
                                                required=False)

            days_shift = sdk2.parameters.Integer('Schedule days shift', required=False, default=14)
            mds_path = sdk2.parameters.String('Path to output file in mds', required=True, default_value='schedule/bus/odnakassa.xml')

        _rasp_scripts_params = use_rasp_scripts_params()
        _email_notification_params = use_email_notification_params()
        _juggler_notification_params = use_juggler_notification_params()

    def on_save(self):
        binary_task.LastBinaryTaskRelease.on_save(self)
        self.add_email_notifications()
        self.add_juggler_notifications(environment=self.Parameters.env_type)

    def on_execute(self):
        binary_task.LastBinaryTaskRelease.on_execute(self)

        self.set_common_environ()

        params = dict(base_url=self.Parameters.base_url,
                      cities=self.Parameters.cities.split(','),
                      start_date=self.Parameters.start_date,
                      days_shift=self.Parameters.days_shift,
                      mds_path=self.Parameters.mds_path)

        from travel.rasp.rasp_scripts.scripts.schedule.odnakassa import run
        run(**{k: v for k, v in params.items() if v})

        log.info('DONE')
