# coding: utf8
from __future__ import absolute_import, division, print_function, unicode_literals

import logging
import os
import tarfile
from distutils.dir_util import copy_tree

from sandbox import common
from sandbox import sdk2
from sandbox.common.types.client import Tag
from sandbox.common.types import task as ctt
from sandbox.projects.common import binary_task
from sandbox.projects.common.ya_deploy import release_integration

from sandbox.projects.rasp.rasp_scripts.GeneratePathfinderData import RaspGeneratePathfinderDataTask
from sandbox.projects.rasp.rasp_scripts.RaspScriptsUpdateQloud import RaspScriptsUpdateQloudMixin
from sandbox.projects.rasp.resource_types import RaspPathfinderRailway, RaspGeometryTesting, RaspGeometryProduction
from sandbox.projects.rasp.utils.email_notifications import EmailNotificationMixin, use_email_notification_params
from sandbox.projects.rasp.utils.juggler import JugglerNotificationMixin, use_juggler_notification_params
from sandbox.projects.rasp.utils.rasp_scripts import RaspScriptsTask, get_resource_ready_params, use_rasp_scripts_params
from sandbox.projects.rasp.resource_types import RaspPathfinderCoreTesting, RaspPathfinderCoreProduction

logger = logging.getLogger()
logger.setLevel(logging.INFO)

GEOMETRY_RES_TTL = 7

GEOMETRY_RES = {
    'testing': RaspGeometryTesting,
    'production': RaspGeometryProduction
}

PATHFINDER_RES = {
    'testing': RaspPathfinderCoreTesting,
    'production': RaspPathfinderCoreProduction
}

PATHFINDER_RAILWAY_RES = RaspPathfinderRailway

LOCAL_PATHFINDER_PATH = 'pathfinder_core_data'
LOCAL_RAILWAY_PATH = 'railway'

LOCAL_RESULT_PICKLE_PATH = 'geometry'
LOCAL_LONG_HAUL_PATH = 'long_haul'


class RaspGenerateGeometryTask(
    release_integration.ReleaseToYaDeployTask2,
    binary_task.LastBinaryTaskRelease,
    sdk2.Task,
    EmailNotificationMixin,
    JugglerNotificationMixin,
    RaspScriptsTask,
    RaspScriptsUpdateQloudMixin
):
    class Requirements(sdk2.Task.Requirements):
        cores = 12
        ram = 10 * 1024
        client_tags = Tag.GENERIC & (Tag.INTEL_E5_2667 | Tag.INTEL_E5_2660V4)

    class Parameters(sdk2.Task.Parameters):
        # geometry approximate execution time - 5h (3 - railway, 1 - limepaths, 1 - long_haul)
        # pathfinder approximate execution time - 30m
        kill_timeout = 3600 * 10

        ext_params = binary_task.binary_release_parameters(stable=True)

        with sdk2.parameters.Group('General') as general:
            generate_pathfinder_data_kill_timeout = 3600 * 2
            update_env_type = sdk2.parameters.String('Update env type', required=True, default='testing')
            qloud_resource_updater__token_owner = sdk2.parameters.String('Token owner', required=True, default='RASP')
            qloud_resource_updater__token_name = sdk2.parameters.String('Token name', required=True, default='rasp-qloud-oauth')

            skip_pathfinder = sdk2.parameters.Bool('Skip pathfinder', default=False, required=True)
            skip_railway = sdk2.parameters.Bool('Skip railway', default=False, required=True)
            skip_limepaths = sdk2.parameters.Bool('Skip limepaths', default=False, required=True)
            skip_long_haul = sdk2.parameters.Bool('Skip long_haul', default=False, required=True)

        _rasp_scripts_params = use_rasp_scripts_params()
        _email_notification_params = use_email_notification_params()
        _juggler_notification_params = use_juggler_notification_params()

    def _run_railway(self, railway_dir, pathfinder_data_path, mif_path):
        from travel.rasp.rasp_scripts.scripts.gen_railway_data import run
        run(railway_dir, pathfinder_data_path, mif_path)

    def _run_limepaths_pack_data(self):
        from travel.rasp.rasp_scripts.scripts.limepaths_pack_data import run
        run()

    def _run_long_haul(self):
        from travel.rasp.rasp_scripts.scripts.long_haul.cache_export import run, upload_dir_mds
        run()

        upload_dir_mds(LOCAL_LONG_HAUL_PATH)

    def on_save(self):
        binary_task.LastBinaryTaskRelease.on_save(self)
        self.add_email_notifications()
        self.add_juggler_notifications(environment=self.Parameters.env_type)

    def on_release(self, additional_parameters):
        super(RaspGenerateGeometryTask, self).on_release(additional_parameters)
        self.mark_released_resources(additional_parameters['release_status'], ttl=GEOMETRY_RES_TTL)

    def on_execute(self):
        if not self.Parameters.skip_pathfinder:
            with self.memoize_stage.generate_pathfinder_data:
                logger.info('Run RaspGeneratePathfinderDataTask')
                task = RaspGeneratePathfinderDataTask(
                    self,
                    owner=self.Parameters.owner,
                    priority=self.Parameters.priority,
                    description='Generate pathfinder data',

                    vault_oauth=self.Parameters.vault_oauth,
                    env_type=self.Parameters.env_type,
                    db_type=self.Parameters.db_type,

                    kill_timeout=self.Parameters.generate_pathfinder_data_kill_timeout,
                    update_env_type=self.Parameters.update_env_type,
                    qloud_resource_updater__token_owner=self.Parameters.qloud_resource_updater__token_owner,
                    qloud_resource_updater__token_name=self.Parameters.qloud_resource_updater__token_name
                )
                task.enqueue()
                raise sdk2.WaitTask([task.id], ctt.Status.Group.FINISH | ctt.Status.Group.BREAK, wait_all=True)

        logger.info('Copy pathfinder data to local path {}'.format(LOCAL_PATHFINDER_PATH))
        pathfinder_res = PATHFINDER_RES.get(self.Parameters.env_type)
        pathfinder_data_path = self._get_resource_path(pathfinder_res, get_resource_ready_params())
        copy_tree(pathfinder_data_path, LOCAL_PATHFINDER_PATH, preserve_mode=0)

        logger.info('Get resources of the previous task run')
        binary_task.LastBinaryTaskRelease.on_execute(self)
        self.set_common_environ()

        resource_params = {'state': common.types.resource.State.READY}

        geometry_res = GEOMETRY_RES.get(self.Parameters.env_type)
        geometry_dir = self._get_resource_path(geometry_res, resource_params)
        copy_tree(geometry_dir, LOCAL_RESULT_PICKLE_PATH, preserve_mode=0)

        railway_path = self._get_resource_path(PATHFINDER_RAILWAY_RES, resource_params)

        # last maps geometry
        mif = sdk2.Resource.find(id=3233839942).first()
        mif_path = self._get_path(mif)

        tar_file = tarfile.open(railway_path)
        tar_file.extractall()

        logger.info('local_dir: {}'.format(os.listdir('.')))
        logger.info('mif: {}'.format(os.listdir(mif_path)))
        logger.info('railway: {}'.format(os.listdir(LOCAL_RAILWAY_PATH)))

        if not self.Parameters.skip_railway:
            logger.info('Generation of the railway data')
            self._run_railway(LOCAL_RAILWAY_PATH, str(self.path(LOCAL_PATHFINDER_PATH)), mif_path)
            logger.info('Generation of the railway data completed')

        if not self.Parameters.skip_limepaths:
            logger.info('Generation of limepaths')
            self._run_limepaths_pack_data()
            logger.info('Generation of limepaths completed')

        if not self.Parameters.skip_railway or not self.Parameters.skip_limepaths:
            logger.info('Upload GEOMETRY resource')
            self._upload_resource(geometry_res, LOCAL_RESULT_PICKLE_PATH)
            logger.info('result_pickle_path: {}'.format(os.listdir(LOCAL_RESULT_PICKLE_PATH)))

        if not self.Parameters.skip_long_haul:
            logger.info('Generation of long_haul')
            self._run_long_haul()
            logger.info('long_haul: {}'.format(os.listdir(LOCAL_LONG_HAUL_PATH)))

        logger.info('GenerateGeometry execution completed')
