# coding: utf8
from __future__ import absolute_import, division, print_function, unicode_literals

import logging
import os

from sandbox import sdk2
from sandbox.common.types.client import Tag
from sandbox.projects.common import binary_task
from sandbox.projects.rasp.rasp_scripts.RaspScriptsUpdateQloud import RaspScriptsUpdateQloudMixin
from sandbox.projects.rasp.utils.juggler import JugglerNotificationMixin, use_juggler_notification_params
from sandbox.projects.rasp.utils.email_notifications import EmailNotificationMixin, use_email_notification_params
from sandbox.projects.rasp.utils.rasp_scripts import RaspScriptsTask, get_resource_ready_params, use_rasp_scripts_params
from sandbox.projects.rasp.utils.solomon import SolomonProgressMixin
from sandbox.projects.rasp.resource_types import RaspGeometryTesting, RaspGeometryProduction

log = logging.getLogger()
log.setLevel(logging.INFO)


GEOMETRY_RES = {
    'testing': RaspGeometryTesting,
    'production': RaspGeometryProduction
}


class RaspGenerateLivemapTask(
    binary_task.LastBinaryTaskRelease,
    sdk2.Task,
    EmailNotificationMixin,
    JugglerNotificationMixin,
    SolomonProgressMixin,
    RaspScriptsTask,
    RaspScriptsUpdateQloudMixin
):
    class Requirements(sdk2.Task.Requirements):
        cores = 1
        ram = 10 * 1024
        client_tags = Tag.GENERIC & Tag.SAS

        class Caches(sdk2.Requirements.Caches):  # enable multislot https://wiki.yandex-team.ru/sandbox/clients/#client-tags-multislot
            pass  # means that task do not use any shared caches

    class Parameters(sdk2.Task.Parameters):
        kill_timeout = 3600 * 5

        ext_params = binary_task.binary_release_parameters(stable=True)

        with sdk2.parameters.Group('General') as general:
            duration = sdk2.parameters.Integer('Duration', required=True, default=20)
            map_generator = sdk2.parameters.String('Map generator', required=True, default='train')

        _rasp_scripts_params = use_rasp_scripts_params()
        _email_notification_params = use_email_notification_params()
        _juggler_notification_params = use_juggler_notification_params()

    def set_task_environ(self):
        geometry_res = GEOMETRY_RES.get(self.Parameters.env_type)
        geometry_dir = self._get_resource_path(geometry_res, get_resource_ready_params())
        log.info('geometry_dir: {}'.format(geometry_dir))

        os.environ['GEOMETRY_BASE_PATH'] = geometry_dir
        os.environ['RASP_MASTER_ONLY'] = 'True'

    def on_save(self):
        binary_task.LastBinaryTaskRelease.on_save(self)
        self.add_email_notifications()
        self.add_juggler_notifications(
            service='{}.{}'.format(self.type, self.Parameters.map_generator),
            environment=self.Parameters.env_type
        )

    def on_execute(self):
        binary_task.LastBinaryTaskRelease.on_execute(self)

        self.set_common_environ()

        from travel.rasp.rasp_scripts.scripts.fill_livemap import run
        with self.report_progress(source=self.Parameters.map_generator, environment=self.Parameters.env_type):
            run(self.Parameters.duration, self.Parameters.map_generator)
