# coding: utf8
from __future__ import absolute_import, division, print_function, unicode_literals

import logging
import os
from distutils.dir_util import copy_tree

from sandbox import sdk2
from sandbox.common.types.client import Tag
from sandbox.common.types import task as ctt
from sandbox.projects.common import binary_task
from sandbox.projects.common.ya_deploy import release_integration
from sandbox.projects.rasp.rasp_scripts.RaspScriptsUpdateQloud import RaspScriptsUpdateQloudMixin
from sandbox.projects.rasp.resource_types import RaspPathfinderCoreTesting, RaspPathfinderCoreProduction
from sandbox.projects.rasp.utils.email_notifications import EmailNotificationMixin, use_email_notification_params
from sandbox.projects.rasp.utils.juggler import JugglerNotificationMixin, use_juggler_notification_params
from sandbox.projects.rasp.utils.rasp_scripts import RaspScriptsTask, get_resource_ready_params, use_rasp_scripts_params

logger = logging.getLogger()
logger.setLevel(logging.INFO)

PATHFINDER_RES_TTL = 4

PATHFINDER_RES = {
    'testing': RaspPathfinderCoreTesting,
    'production': RaspPathfinderCoreProduction
}

LOCAL_PATHFINDER_PATH = 'pathfinder_core_data'


class RaspGeneratePathfinderDataTask(
    release_integration.ReleaseToNannyAndYaDeployTask2,
    binary_task.LastBinaryTaskRelease,
    sdk2.Task,
    EmailNotificationMixin,
    JugglerNotificationMixin,
    RaspScriptsTask,
    RaspScriptsUpdateQloudMixin
):
    class Requirements(sdk2.Task.Requirements):
        cores = 12
        ram = 10 * 1024
        client_tags = Tag.GENERIC & (Tag.INTEL_E5_2667 | Tag.INTEL_E5_2660V4)

        class Caches(sdk2.Requirements.Caches):  # enable multislot https://wiki.yandex-team.ru/sandbox/clients/#client-tags-multislot
            pass  # means that task do not use any shared caches

    class Parameters(sdk2.Task.Parameters):
        # pathfinder approximate execution time - 30m
        kill_timeout = 3600 * 2

        ext_params = binary_task.binary_release_parameters(stable=True)

        with sdk2.parameters.Group('General') as general:
            update_env_type = sdk2.parameters.String('Update env type', required=True, default='testing')
            qloud_resource_updater__token_owner = sdk2.parameters.String('Token owner', required=True, default='RASP')
            qloud_resource_updater__token_name = sdk2.parameters.String('Token name', required=True, default='rasp-qloud-oauth')

        _rasp_scripts_params = use_rasp_scripts_params()
        _email_notification_params = use_email_notification_params()
        _juggler_notification_params = use_juggler_notification_params()

    def _run_pathfinder_price(self):
        from travel.rasp.rasp_scripts.scripts.pathfinder.min_prices_from_logs.dump_yt_price_to_pathfinder import run
        run()

    def _run_pathfinder(self):
        from travel.rasp.rasp_scripts.scripts.pathfinder.gen_pathfinder_data import run
        run()

    def on_save(self):
        binary_task.LastBinaryTaskRelease.on_save(self)
        self.add_email_notifications()
        self.add_juggler_notifications(environment=self.Parameters.env_type)

    def on_release(self, additional_parameters):
        super(RaspGeneratePathfinderDataTask, self).on_release(additional_parameters)
        self.mark_released_resources(additional_parameters['release_status'], ttl=PATHFINDER_RES_TTL)

    def on_enqueue(self):
        semaphore_name = '{}_for_{}'.format(self.type.name, self.Parameters.update_env_type)
        self.Requirements.semaphores = ctt.Semaphores(
            acquires=[
                ctt.Semaphores.Acquire(name=semaphore_name, capacity=1)
            ]
        )

    def on_execute(self):
        binary_task.LastBinaryTaskRelease.on_execute(self)

        self.set_common_environ()

        pathfinder_res = PATHFINDER_RES.get(self.Parameters.env_type)
        pathfinder_data_path = self._get_resource_path(pathfinder_res, get_resource_ready_params())
        copy_tree(pathfinder_data_path, LOCAL_PATHFINDER_PATH, preserve_mode=0)

        logger.info('local_dir: {}'.format(os.listdir('.')))
        logger.info('pathfinder: {}'.format(os.listdir(LOCAL_PATHFINDER_PATH)))

        logger.info('Generation of the data for pathfinder')
        self._run_pathfinder()
        logger.info('Generation of the prices for pathfinder')
        self._run_pathfinder_price()
        logger.info('Upload resource')
        self._upload_resource(pathfinder_res, LOCAL_PATHFINDER_PATH)
        logger.info('Pathfinder data processing completed')
