# coding: utf8
from __future__ import absolute_import, division, print_function, unicode_literals

import logging

from sandbox import sdk2
from sandbox.projects.common import binary_task

from sandbox.projects.rasp.RaspBuildRaspDataAndUpdate import RaspBuildRaspDataAndUpdate
from sandbox.projects.rasp.TrainsDumpActualDirections import TrainsDumpActualDirections
from sandbox.projects.rasp.TrainsGenerateSitemap import TrainsGenerateSitemap
from sandbox.projects.rasp.infra.RunAndRelease import RaspRunAndReleaseTask
from sandbox.projects.rasp.rasp_scripts.RaspApiPublicStationsList import RaspApiPublicStationsList
from sandbox.projects.rasp.utils.email_notifications import EmailNotificationMixin, use_email_notification_params
from sandbox.projects.rasp.utils.juggler import JugglerNotificationMixin, use_juggler_notification_params
from sandbox.projects.rasp.utils.rasp_scripts import RaspScriptsTask, use_rasp_scripts_params

log = logging.getLogger(__name__)
log.setLevel(logging.INFO)


class RaspAdminScriptAfterRun(
    binary_task.LastBinaryTaskRelease,
    sdk2.Task,
    EmailNotificationMixin,
    JugglerNotificationMixin,
    RaspScriptsTask
):
    class Requirements(sdk2.Task.Requirements):
        cores = 1
        ram = 1 * 1024

        class Caches(sdk2.Requirements.Caches):  # enable multislot https://wiki.yandex-team.ru/sandbox/clients/#client-tags-multislot
            pass  # means that task do not use any shared caches

    class Parameters(sdk2.Task.Parameters):
        kill_timeout = 3600

        with sdk2.parameters.Group('MySQL'):
            run_rasp_build_rasp_data_and_update = sdk2.parameters.Bool('[deprecated]', default=True)
            mysql_host = sdk2.parameters.String('MySQL host name')
            mysql_user = sdk2.parameters.String('MySQL user')
            mysql_db = sdk2.parameters.String('MySQL database name')

        with sdk2.parameters.Group('General') as general:
            script_name = sdk2.parameters.String(
                'Name of script executed in rasp admin',
                required=True
            )

        ext_params = binary_task.binary_release_parameters(stable=True)
        _rasp_scripts_params = use_rasp_scripts_params()
        _email_notification_params = use_email_notification_params()
        _juggler_notification_params = use_juggler_notification_params()

    def on_save(self):
        binary_task.LastBinaryTaskRelease.on_save(self)
        self.add_email_notifications()
        self.add_juggler_notifications(environment=self.Parameters.env_type)

    def on_execute(self):
        self.set_common_environ()

        from common.data_api.sandbox.admin_script_after_run import AdminScripts
        from common.settings.db import WorkInstance

        script_name = self.Parameters.script_name
        db_type = self.Parameters.db_type
        tasks = []

        if (
            script_name == AdminScripts.SWITCH_BASES
            or (script_name == AdminScripts.PREPARE_ALL and db_type == WorkInstance.code)
        ):
            tasks.append(RaspApiPublicStationsList(
                self,
                owner=self.owner,
                description=self.Parameters.description,
                env_type=self.Parameters.env_type,
                db_type=WorkInstance.code
            ))

            # tasks.append(RaspInfocenterTask(
            #     self,
            #     __requirements__={'cores': 8, 'ram': 2 * 64 * 1024, 'disk_space': 100 * 1024},
            #     owner=self.owner,
            #     description=self.Parameters.description,
            #     env_type=self.Parameters.env_type,
            #     db_type=WorkInstance.code,
            #     source='run_subscription_changes_after_prepare_all',
            #     priority=Priority(Priority.Class.SERVICE, Priority.Subclass.HIGH)
            # ))

            tasks.append(TrainsDumpActualDirections(
                self,
                owner=self.owner,
                description=self.Parameters.description,
                env_type=self.Parameters.env_type,
                db_type=WorkInstance.code
            ))

        if script_name == AdminScripts.SWITCH_BASES:
            tasks.append(TrainsGenerateSitemap(
                self,
                owner=self.owner,
                description=self.Parameters.description,
                env_type=self.Parameters.env_type,
                db_type=self.Parameters.db_type,
                check_sitemaps=False,
            ))

            tasks.append(RaspRunAndReleaseTask(
                self,
                owner=self.owner,
                description=self.Parameters.description,
                task_name='RaspUpdateMaintenanceConf',
                task_path='sandbox.projects.rasp.rasp_scripts.RaspUpdateMaintenanceConf',
                task_params={
                    'env_type': self.Parameters.env_type,
                    'db_type': self.Parameters.db_type
                },
                env_type=self.Parameters.env_type,
                db_type=self.Parameters.db_type
            ))

        if script_name == AdminScripts.PREPARE_ALL and db_type == WorkInstance.code:
            tasks.append(RaspBuildRaspDataAndUpdate(
                self,
                owner=self.owner,
                description='Build new rasp data and update apps',
                rasp_data_maker__maker_host=self.Parameters.mysql_host,
                rasp_data_maker__maker_user=self.Parameters.mysql_user,
                rasp_data_maker__maker_database_name=self.Parameters.mysql_db,
                rasp_data_maker__maker_environment=self.Parameters.env_type,
                rasp_data_maker__maker_build_common_dicts_archive=True,

                release_on_ready=True
            ))

        for task in tasks:
            task.enqueue()
