# coding: utf8
from __future__ import absolute_import, division, print_function, unicode_literals

import os

from sandbox import sdk2
from sandbox.projects.common.binary_task import LastBinaryTaskRelease, binary_release_parameters

from sandbox.projects.rasp.utils.email_notifications import EmailNotificationMixin, use_email_notification_params
from sandbox.projects.rasp.utils.juggler import JugglerNotificationMixin, use_juggler_notification_params
from sandbox.projects.rasp.utils.rasp_scripts import RaspScriptsTask, use_rasp_scripts_params
from sandbox.projects.rasp.utils.solomon import SolomonProgressMixin


class RaspCleanTemporarySchedule(
    sdk2.Task,
    RaspScriptsTask,
    LastBinaryTaskRelease,
    SolomonProgressMixin,
    JugglerNotificationMixin,
    EmailNotificationMixin,
):
    class Requirements(sdk2.Task.Requirements):
        cores = 1
        ram = 1024 * 10

        class Caches(sdk2.Requirements.Caches):  # https://wiki.yandex-team.ru/sandbox/clients/#client-tags-multislot
            pass  # task does not use any shared caches

    class Parameters(sdk2.Task.Parameters):
        kill_timeout = 3600
        ext_params = binary_release_parameters(stable=True)

        mds_prefix = sdk2.parameters.String('Mds prefix', default=None)
        past_days = sdk2.parameters.Integer('Past days', default=None)

        _rasp_scripts_params = use_rasp_scripts_params()
        _email_notification_params = use_email_notification_params()
        _juggler_notification_params = use_juggler_notification_params()

    def on_save(self):
        LastBinaryTaskRelease.on_save(self)

        if self.Parameters.mds_prefix and 'schedule-temporary' not in self.Parameters.mds_prefix:
            raise Exception('schedule-temporary not in mds_prefix')

        self.add_juggler_notifications(environment=self.Parameters.env_type)
        self.add_email_notifications()

    def on_execute(self):
        LastBinaryTaskRelease.on_execute(self)

        self.set_common_environ()

        mds_prefix = self.Parameters.mds_prefix if self.Parameters.mds_prefix else None

        from travel.rasp.rasp_scripts.scripts.schedule.clean_temporary_schedule import run

        with self.report_progress('clean_temporary_schedule', self.Parameters.env_type):
            run(mds_prefix=mds_prefix, past_days=self.Parameters.past_days)
