# coding: utf8
from __future__ import absolute_import, division, print_function, unicode_literals

import logging
import shutil

from sandbox import sdk2
from sandbox.common.types.task import ReleaseStatus, Status
from sandbox.projects.common.build import parameters as build_parameters
from sandbox.projects.rasp.utils.email_notifications import EmailNotificationMixin, use_email_notification_params
from sandbox.projects.sandbox.deploy_binary_task import DeployBinaryTask


log = logging.getLogger(__name__)
log.setLevel(logging.INFO)


class RaspDeployBinaryTask(sdk2.Task, EmailNotificationMixin):
    class Requirements(sdk2.Task.Requirements):
        class Caches(sdk2.Requirements.Caches):  # enable multislot https://wiki.yandex-team.ru/sandbox/clients/#client-tags-multislot
            pass  # means that task do not use any shared caches

        cores = 1  # cores number is required to enable multislot
        ram = 1 * 1024

    class Parameters(sdk2.Task.Parameters):
        kill_timeout = 600

        with sdk2.parameters.Group('General') as general:
            task_type = sdk2.parameters.String(
                'Name of the task type',
                default='RASP_SCRIPT_EXAMPLE',
                required=True
            )
            target = sdk2.parameters.String(
                'Path to target task relative to Arcadia root',
                default='sandbox/projects/rasp/rasp_scripts/RaspScriptExample/RaspScriptExample',
                required=True
            )
            yt_token_vault = sdk2.parameters.Vault(
                'YT token vault entry (owner:name, or only name)',
                default='YT_TOKEN'
            )
            run_testing_release = sdk2.parameters.Bool(
                'Release task binary into testing',
                default=True
            )
            arcadia_patch = build_parameters.ArcadiaPatch(group=None, default='')
            build_task_container_resource = sdk2.parameters.Resource(
                'Run build task using LXC image stored in specified resource', required=False
            )

        _email_notification_params = use_email_notification_params()

    def on_execute(self):
        deploy_task_id = self.Context.deploy_task_id
        if not deploy_task_id:
            self.run_deploy_task()

        deploy_task = sdk2.Task.find(id=deploy_task_id).first()
        logging.info('DEPLOY_BINARY_TASK status=%s', deploy_task.status)

        resource = sdk2.service_resources.SandboxTasksBinary.find(task=deploy_task).first()
        self.clone_task_binary(resource, ('task_type', 'commit_revision', 'binary_hash', 'binary_age'))

        if self.Parameters.run_testing_release:
            logging.info('Releasing DEPLOY_BINARY_TASK %d into testing', deploy_task_id)
            self.server.release(
                task_id=deploy_task_id,
                type=ReleaseStatus.TESTING,
                subject='Release'
            )

    def clone_task_binary(self, scr, copy_attributes):
        dst = sdk2.ResourceData(scr.type(
            task=self,
            description=scr.description,
            path=scr.path.name,
            **{a: getattr(scr, a) for a in copy_attributes if hasattr(scr, a)}
        ))

        src_path = sdk2.ResourceData(scr).path
        shutil.copy(str(src_path), str(dst.path))
        dst.ready()

        logging.info('Clone SANDBOX_TASKS_BINARY from resource %d', scr.id)

    def run_deploy_task(self):
        deploy_task = DeployBinaryTask(
            self,
            __requirements__={'container_resource': self.Parameters.build_task_container_resource},
            owner=self.Parameters.owner,
            description='Release {}'.format(self.Parameters.task_type),
            priority=self.Parameters.priority,
            target=self.Parameters.target,
            yt_token_vault=self.Parameters.yt_token_vault,
            arcadia_patch=self.Parameters.arcadia_patch,
            attrs={'task_type': self.Parameters.task_type}
        )

        deploy_task.enqueue()
        logging.info('Created DEPLOY_BINARY_TASK id=%d', deploy_task.id)

        self.Context.deploy_task_id = deploy_task.id

        raise sdk2.WaitTask(
            [deploy_task],
            Status.Group.FINISH | Status.Group.BREAK,
            wait_all=True
        )

    def on_save(self):
        super(RaspDeployBinaryTask, self).on_save()
        self.add_email_notifications()
