# coding: utf8
from __future__ import absolute_import, division, print_function, unicode_literals

import logging

from sandbox import sdk2
from sandbox.common.types.misc import DnsType
from sandbox.projects.common import binary_task
from sandbox.projects.rasp.utils.email_notifications import EmailNotificationMixin, use_email_notification_params
from sandbox.projects.rasp.utils.juggler import JugglerNotificationMixin, use_juggler_notification_params
from sandbox.projects.rasp.utils.rasp_scripts import RaspScriptsTask, use_rasp_scripts_params

log = logging.getLogger(__name__)
log.setLevel(logging.INFO)


class RaspGenericTask(
    binary_task.LastBinaryTaskRelease,
    sdk2.Task,
    EmailNotificationMixin,
    JugglerNotificationMixin,
    RaspScriptsTask
):
    """
    Таска, умеющая импортировать и запускать функцию из rasp_scripts по заданному пути импорта.
    Можно использовать для скриптов, не требующих особых настроек.

    Пример:
      func_import_path = 'script_example.run'
      func_args = ['a', 'b']
      func_kwargs = {'my': 'vaue'}
    Запустится:
      travel.rasp.rasp_scripts.scripts.script_example.run('a', 'b', my='value')
    """

    SCRIPTS_BASE_PATH = "travel.rasp.rasp_scripts.scripts."

    class Requirements(sdk2.Task.Requirements):
        cores = 1
        ram = 1 * 1024
        dns = DnsType.DNS64

        class Caches(sdk2.Requirements.Caches):  # enable multislot https://wiki.yandex-team.ru/sandbox/clients/#client-tags-multislot
            pass  # means that task do not use any shared caches

    class Parameters(sdk2.Task.Parameters):
        kill_timeout = 600

        ext_params = binary_task.binary_release_parameters(stable=True)

        with sdk2.parameters.Group('General') as general:
            func_import_path = sdk2.parameters.String('Dotted import path to function', required=True)
            func_args = sdk2.parameters.List('List of str args, passed to func', required=False)
            func_kwargs = sdk2.parameters.Dict('Dict of str kwargs, passed to func', required=False)

        _rasp_scripts_params = use_rasp_scripts_params()
        _email_notification_params = use_email_notification_params()
        _juggler_notification_params = use_juggler_notification_params()

    def on_save(self):
        binary_task.LastBinaryTaskRelease.on_save(self)
        self.add_email_notifications()
        self.add_juggler_notifications(
            service='{}.{}'.format(self.type, self.Parameters.func_import_path),
            environment=self.Parameters.env_type
        )

    def on_execute(self):
        binary_task.LastBinaryTaskRelease.on_execute(self)

        self.Parameters.description = self.Parameters.description + " #Run: {} ({}, {})".format(
            self.Parameters.func_import_path,
            self.Parameters.func_args,
            self.Parameters.func_kwargs,
        )

        self.set_common_environ()

        from django.utils.module_loading import import_string

        func_import_path_full = self.SCRIPTS_BASE_PATH + self.Parameters.func_import_path
        log.info("Importing func by path: %s", self.Parameters.func_import_path)
        func = import_string(func_import_path_full)
        log.info("Running func %s", func)
        func(*self.Parameters.func_args, **self.Parameters.func_kwargs)
