# coding: utf8
from __future__ import absolute_import, division, print_function, unicode_literals

import os
import logging

from sandbox import sdk2
from sandbox.projects.common import binary_task
from sandbox.projects.rasp.utils.email_notifications import EmailNotificationMixin, use_email_notification_params
from sandbox.projects.rasp.utils.juggler import JugglerNotificationMixin, use_juggler_notification_params
from sandbox.projects.rasp.utils.rasp_scripts import RaspScriptsTask, use_rasp_scripts_params
from sandbox.projects.rasp.utils.solomon import SolomonProgressMixin


log = logging.getLogger(__name__)
log.setLevel(logging.INFO)


class RaspInfocenterTask(
    binary_task.LastBinaryTaskRelease,
    sdk2.Task,
    EmailNotificationMixin,
    JugglerNotificationMixin,
    RaspScriptsTask,
    SolomonProgressMixin,
):
    class Requirements(sdk2.Task.Requirements):
        cores = 1
        ram = 1 * 1024

        class Caches(sdk2.Requirements.Caches):  # enable multislot https://wiki.yandex-team.ru/sandbox/clients/#client-tags-multislot
            pass  # means that task do not use any shared caches

    class Parameters(sdk2.Task.Parameters):
        kill_timeout = 3600

        ext_params = binary_task.binary_release_parameters(stable=True)

        with sdk2.parameters.Group('General') as general:
            source = sdk2.parameters.String('Script name', required=True, choices=[
                (a, a) for a in (
                    'run_subscription_changes', 'run_subscription_changes_after_prepare_all',
                    'clean_subscription_changes', 'monitoring'
                )
            ])

        _rasp_scripts_params = use_rasp_scripts_params()
        _email_notification_params = use_email_notification_params()
        _juggler_notification_params = use_juggler_notification_params()

    def on_save(self):
        binary_task.LastBinaryTaskRelease.on_save(self)
        self.add_email_notifications()
        self.add_juggler_notifications(
            service='{}.{}'.format(self.type, self.Parameters.source),
            environment=self.Parameters.env_type
        )

    def set_task_environ(self):
        os.environ['DJANGO_SETTINGS_MODULE'] = 'travel.rasp.info_center.bin.sandbox.local_settings'

    def on_execute(self):
        self.set_common_environ()

        from travel.rasp.info_center.info_center.scripts import (
            clean_subscription_changes, run_subscription_changes, run_subscription_changes_after_prepare_all
        )
        from travel.rasp.info_center.info_center.scripts.monitoring import notification

        scripts = {
            'run_subscription_changes': run_subscription_changes.run,
            'run_subscription_changes_after_prepare_all': run_subscription_changes_after_prepare_all.run,
            'clean_subscription_changes': clean_subscription_changes.run,
            'monitoring': notification.run
        }

        with self.report_progress(source=self.Parameters.source, environment=self.Parameters.env_type):
            scripts[self.Parameters.source]()
