# coding: utf8
from __future__ import absolute_import, division, print_function, unicode_literals

import logging
import os

from sandbox import sdk2
from sandbox.common.types.client import Tag
from sandbox.projects.common import binary_task
from sandbox.projects.rasp.utils.email_notifications import EmailNotificationMixin, use_email_notification_params
from sandbox.projects.rasp.utils.geobase import GeobaseMixin, use_geobase_params
from sandbox.projects.rasp.utils.juggler import JugglerNotificationMixin, use_juggler_notification_params
from sandbox.projects.rasp.utils.rasp_scripts import RaspScriptsTask, use_rasp_scripts_params
from sandbox.projects.rasp.utils.solomon import SolomonProgressMixin

log = logging.getLogger(__name__)
log.setLevel(logging.INFO)


class RaspNearestSuburban(
    binary_task.LastBinaryTaskRelease,
    sdk2.Task,
    EmailNotificationMixin,
    JugglerNotificationMixin,
    RaspScriptsTask,
    SolomonProgressMixin,
    GeobaseMixin
):
    class Requirements(sdk2.Task.Requirements):
        cores = 1
        ram = 1 * 1024
        client_tags = Tag.SAS | Tag.VLA

        class Caches(sdk2.Requirements.Caches):  # enable multislot https://wiki.yandex-team.ru/sandbox/clients/#client-tags-multislot
            pass  # means that task do not use any shared caches

    class Parameters(sdk2.Task.Parameters):
        kill_timeout = 3600

        ext_params = binary_task.binary_release_parameters(stable=True)

        with sdk2.parameters.Group('General') as general:
            source = sdk2.parameters.String('Script name', required=True, choices=[(a,a) for a in
                ('update_all_nearest_suburbans', 'update_nearest_suburbans', 'update_nearest_main_stations_suburbans')
            ])

        _rasp_scripts_params = use_rasp_scripts_params()
        _email_notification_params = use_email_notification_params()
        _juggler_notification_params = use_juggler_notification_params()
        _geobase_params = use_geobase_params()

    def on_save(self):
        binary_task.LastBinaryTaskRelease.on_save(self)
        self.add_email_notifications()
        self.add_juggler_notifications(
            service='{}.{}'.format(self.type, self.Parameters.source),
            environment=self.Parameters.env_type
        )

    def set_task_environ(self):
        os.environ['RASP_GEOBASE_DATA_PATH'] = self.geobase_path

    def on_execute(self):
        self.download_geobase()
        self.set_common_environ()

        from travel.rasp.tasks.nearest_suburbans.main_stations_generator import MainStationsSuburbansGenerator
        from travel.rasp.tasks.nearest_suburbans.nearest_generator import NearestSuburbansGenerator

        scripts = {
            'update_all_nearest_suburbans': NearestSuburbansGenerator(False, log).run_script,
            'update_nearest_suburbans': NearestSuburbansGenerator(True, log).run_script,
            'update_nearest_main_stations_suburbans': MainStationsSuburbansGenerator(log).run_script
        }

        with self.report_progress(source=self.Parameters.source, environment=self.Parameters.env_type):
            scripts[self.Parameters.source]()
