# coding: utf8
from __future__ import absolute_import, division, print_function, unicode_literals

import logging
import os

from sandbox import sdk2
from sandbox.projects.common import binary_task
from sandbox.projects.rasp.utils.email_notifications import EmailNotificationMixin, use_email_notification_params
from sandbox.projects.rasp.utils.juggler import JugglerNotificationMixin, use_juggler_notification_params
from sandbox.projects.rasp.utils.rasp_scripts import RaspScriptsTask, use_rasp_scripts_params
from sandbox.projects.rasp.utils.solomon import SolomonProgressMixin

log = logging.getLogger(__name__)
log.setLevel(logging.INFO)

LXC_CONTAINER_WITH_IBM_DRIVER = 2270325684


class RaspRzdTask(
    binary_task.LastBinaryTaskRelease,
    sdk2.Task,
    EmailNotificationMixin,
    JugglerNotificationMixin,
    RaspScriptsTask,
    SolomonProgressMixin
):
    class Requirements(sdk2.Task.Requirements):
        container_resource = LXC_CONTAINER_WITH_IBM_DRIVER
        cores = 1
        ram = 1 * 1024

        class Caches(sdk2.Requirements.Caches):  # enable multislot https://wiki.yandex-team.ru/sandbox/clients/#client-tags-multislot
            pass  # means that task do not use any shared caches

    class Parameters(sdk2.Task.Parameters):
        kill_timeout = 3600 * 2

        ext_params = binary_task.binary_release_parameters(stable=True)

        with sdk2.parameters.Group('General') as general:
            source = sdk2.parameters.String('Rzd script to be executed', required=True, choices=[
                (a, a) for a in (
                    'fetch_events_from_rzd', 'fetch_events_from_sqs', 'update_changes', 'update_full', 'monitoring'
                )
            ])
            action = sdk2.parameters.String('Script action', required=False, default=None)

        _rasp_scripts_params = use_rasp_scripts_params()
        _email_notification_params = use_email_notification_params()
        _juggler_notification_params = use_juggler_notification_params()

    def on_save(self):
        binary_task.LastBinaryTaskRelease.on_save(self)
        self.add_email_notifications()
        self.add_juggler_notifications(
            service='{}.{}'.format(self.type, self.Parameters.source),
            environment=self.Parameters.env_type
        )

    def set_task_environ(self):
        os.environ['DJANGO_SETTINGS_MODULE'] = 'travel.rasp.suburban_tasks.bin.sandbox.local_settings'

    def on_execute(self):
        self.set_common_environ()

        from travel.rasp.suburban_tasks.suburban_tasks.scripts import (
            fetch_events_from_rzd, fetch_events_from_sqs, update_changes, update_full, monitoring
        )

        suburban_tasks = {
            'fetch_events_from_rzd': fetch_events_from_rzd.run,
            'fetch_events_from_sqs': fetch_events_from_sqs.run,
            'update_changes': update_changes.run,
            'update_full': update_full.run,
            'monitoring': monitoring.run
        }

        task = suburban_tasks[self.Parameters.source]

        with self.report_progress(self.Parameters.source, self.Parameters.env_type):
            if self.Parameters.action:
                task(action=self.Parameters.action)
            else:
                task()
