# coding: utf8
from __future__ import absolute_import, division, print_function, unicode_literals

import logging

from sandbox import sdk2
from sandbox.common.types.misc import DnsType
from sandbox.projects.common import binary_task
from sandbox.projects.rasp.utils.rasp_scripts import RaspScriptsTask, use_rasp_scripts_params
from sandbox.projects.resource_types import OTHER_RESOURCE

# в этот момент еще не инициализирован никакой расписанческий код -
# джанги нет, настроек логгинга нет - голый Sandbox.
# Настраиваем логирование сами
log = logging.getLogger(__name__)
log.setLevel(logging.INFO)


class RaspScriptExample(
    binary_task.LastBinaryTaskRelease,
    sdk2.Task,
    RaspScriptsTask
):
    class Requirements(sdk2.Task.Requirements):
        class Caches(sdk2.Requirements.Caches):  # enable multislot https://wiki.yandex-team.ru/sandbox/clients/#client-tags-multislot
            pass  # means that task do not use any shared caches

        cores = 1  # cores number is required to enable multislot
        ram = 1 * 1024
        dns = DnsType.DNS64

    class Parameters(sdk2.Task.Parameters):
        kill_timeout = 600

        ext_params = binary_task.binary_release_parameters(stable=True)  # нужно для работы LastBinaryTaskRelease

        with sdk2.parameters.Group('General') as general:
            some_string = sdk2.parameters.String('My url', required=False, default='https://rasp.yandex.ru')
            some_int = sdk2.parameters.Integer('My integer', required=True, default=42)
            some_dict = sdk2.parameters.Dict('any kv', required=False, default={'key42': 'value42'})
            some_list = sdk2.parameters.List('list of something', required=False, default=['my', 'strange', 'list'])

        _rasp_scripts_params = use_rasp_scripts_params()

    def on_save(self):
        binary_task.LastBinaryTaskRelease.on_save(self)  # нужно для работы LastBinaryTaskRelease
        self.Parameters.notifications = None  # выключаем нотификации - для этого таска они не нужны

    def on_execute(self):
        """
        Код внутри sandbox-таски должен быть самым минимальным - какая-то базовая инициализация и всё.
        Вся логика скрипта должна быть независима от sandbox, насколько это возможно.
        Это сильно упрощает разработку, а также позволяет запускать код не только из sandbox, но и в любом другом месте
        """

        # нужно для работы LastBinaryTaskRelease
        binary_task.LastBinaryTaskRelease.on_execute(self)

        log.info('RaspScriptExample.on_execute start')
        log.info('some_string %s', self.Parameters.some_string)
        log.info('some_int %s', self.Parameters.some_int)
        log.info('some_dict %s', self.Parameters.some_dict)
        log.info('some_list %s', self.Parameters.some_list)

        # базовая инициализация настроек для расписанческого кода
        self.set_common_environ()

        # запускаем какой-нибуь наш скрипт
        from travel.rasp.rasp_scripts.scripts import script_example
        out_file_path = script_example.run()

        # сохраняем данные из скрипта в ресурс.
        # OTHER_RESOURCE - только для примера, лучше заводить отделные типы для ресурсов
        resource = sdk2.ResourceData(OTHER_RESOURCE(self, 'my result resource', out_file_path))
        resource.ready()

        log.info('RaspScriptExample.on_execute done')
