# coding: utf8
from __future__ import absolute_import, division, print_function, unicode_literals

import logging
import os

from sandbox import common
from sandbox import sdk2
from sandbox.common.types.misc import DnsType
from sandbox.projects.common import binary_task

from sandbox.projects.rasp.rasp_scripts.RaspScriptsUpdateQloud import RaspScriptsUpdateQloudMixin
from sandbox.projects.rasp.resource_types import (
    MovistaBillingCompareTesting, MovistaBillingCompareProduction, ImBillingCompareTesting, ImBillingCompareProduction
)
from sandbox.projects.rasp.utils.email_notifications import EmailNotificationMixin, use_email_notification_params
from sandbox.projects.rasp.utils.juggler import JugglerNotificationMixin, use_juggler_notification_params
from sandbox.projects.rasp.utils.rasp_scripts import RaspScriptsTask, use_rasp_scripts_params
from sandbox.projects.rasp.utils.solomon import SolomonProgressMixin

logger = logging.getLogger()
logger.setLevel(logging.INFO)


BILLING_COMPARE_RES = {
    ('movista', 'testing'): MovistaBillingCompareTesting,
    ('movista', 'production'): MovistaBillingCompareProduction,
    ('im', 'testing'): ImBillingCompareTesting,
    ('im', 'production'): ImBillingCompareProduction
}

LOCAL_COMPARE_RESULT_PATH = 'suburban_billing_compare'
PROVIDERS = ('movista', 'im')
COMPARE_PERIODS = ('custom_period', 'previous_day', 'previous_day_month', 'previous_month')

REPORT_FORMAT = (
    '<a href="https://sandbox.yandex-team.ru/resource/{id}/view">Resource</a> and '
    '<a href="https://proxy.sandbox.yandex-team.ru/{id}/result?force_text_mode=1">report</a>'
)


class RaspSuburbanCompareWithBilling(
    binary_task.LastBinaryTaskRelease,
    sdk2.Task,
    EmailNotificationMixin,
    JugglerNotificationMixin,
    RaspScriptsTask,
    RaspScriptsUpdateQloudMixin,
    SolomonProgressMixin
):
    class Requirements(sdk2.Task.Requirements):
        cores = 1
        ram = 1 * 1024
        dns = DnsType.DNS64

        class Caches(sdk2.Requirements.Caches):  # enable multislot https://wiki.yandex-team.ru/sandbox/clients/#client-tags-multislot
            pass  # means that task do not use any shared caches

    class Parameters(sdk2.Task.Parameters):
        kill_timeout = 3600
        ext_params = binary_task.binary_release_parameters(stable=True)

        with sdk2.parameters.Group('General') as general:
            provider = sdk2.parameters.String('Provider', required=True, choices=[
                (a, a) for a in PROVIDERS
            ])
            compare_period = sdk2.parameters.String('Comparing period', required=True, choices=[
                (a, a) for a in COMPARE_PERIODS
            ])
            compare_start_date = sdk2.parameters.String('Comparing custom start date', required=False)
            compare_end_date = sdk2.parameters.String('Comparing custom end date', required=False)

        _rasp_scripts_params = use_rasp_scripts_params()
        _email_notification_params = use_email_notification_params()
        _juggler_notification_params = use_juggler_notification_params()

    def on_save(self):
        binary_task.LastBinaryTaskRelease.on_save(self)
        self.add_email_notifications()
        self.add_juggler_notifications(
            service='{}.{}'.format(self.type, self.Parameters.provider),
            environment=self.Parameters.env_type
        )

    def on_execute(self):
        binary_task.LastBinaryTaskRelease.on_execute(self)
        self.set_common_environ()

        from travel.rasp.rasp_scripts.scripts.suburban import compare_provider_report_with_billing

        with self.report_progress(
            source='compare_{}_with_billing'.format(self.Parameters.provider), environment=self.Parameters.env_type
        ):
            compare_res = BILLING_COMPARE_RES.get((self.Parameters.provider, self.Parameters.env_type))

            logger.info('Run comparing script. Provider: {}, Period: {}'.format(
                self.Parameters.provider, self.Parameters.compare_period
            ))
            has_errors = compare_provider_report_with_billing.run(
                self.Parameters.provider,
                self.Parameters.compare_period,
                self.Parameters.compare_start_date,
                self.Parameters.compare_end_date
            )

            logger.info('local_dir: {}'.format(os.listdir('.')))
            logger.info('suburban_billing_compare: {}'.format(os.listdir(LOCAL_COMPARE_RESULT_PATH)))

            logger.info('Upload result resource')
            self._upload_resource(compare_res, LOCAL_COMPARE_RESULT_PATH)
            resource_id = self.Context.resource_suburban_billing_compare_id

            if has_errors:
                self.set_info(REPORT_FORMAT.format(id=resource_id), do_escape=False)
                raise common.errors.TaskFailure('Difference between {} and billing orders are found'.format(
                    self.Parameters.provider
                ))
