# coding: utf8
from __future__ import absolute_import, division, print_function, unicode_literals

from sandbox import sdk2
from sandbox.projects.common import binary_task
from sandbox.projects.rasp.utils.email_notifications import EmailNotificationMixin, use_email_notification_params
from sandbox.projects.rasp.utils.juggler import JugglerNotificationMixin, use_juggler_notification_params
from sandbox.projects.rasp.utils.rasp_scripts import RaspScriptsTask, use_rasp_scripts_params
from sandbox.projects.rasp.utils.solomon import SolomonProgressMixin


class RaspSuburbanEvents(
    binary_task.LastBinaryTaskRelease,
    sdk2.Task,
    EmailNotificationMixin,
    JugglerNotificationMixin,
    RaspScriptsTask,
    SolomonProgressMixin
):
    class Requirements(sdk2.Task.Requirements):
        cores = 1
        ram = 1 * 1024

        class Caches(sdk2.Requirements.Caches):  # enable multislot https://wiki.yandex-team.ru/sandbox/clients/#client-tags-multislot
            pass  # means that task do not use any shared caches

    class Parameters(sdk2.Task.Parameters):
        kill_timeout = 3600

        ext_params = binary_task.binary_release_parameters(stable=True)

        with sdk2.parameters.Group('General') as general:
            source = sdk2.parameters.String('Script name', required=True, choices=[
                (a, a) for a in (
                    'monitoring',
                    'update_suburban_events',
                    'update_expected_suburban_events',
                    'update_expected_suburban_events_after_prepare_all',
                    'clean_suburban_events',
                    'update_company_crashes',
                    'update_movista_raw_cancels',
                    'update_movista_matched_cancels'
                )
            ])

        _rasp_scripts_params = use_rasp_scripts_params()
        _email_notification_params = use_email_notification_params()
        _juggler_notification_params = use_juggler_notification_params()

    def on_save(self):
        binary_task.LastBinaryTaskRelease.on_save(self)
        self.add_email_notifications()
        self.add_juggler_notifications(
            service='{}.{}'.format(self.type, self.Parameters.source),
            environment=self.Parameters.env_type
        )

    def on_execute(self):
        self.set_common_environ()

        from travel.rasp.tasks.monitoring import suburban_events
        from travel.rasp.tasks.suburban_events import clean_suburban_events
        from travel.rasp.tasks.suburban_events import update_company_crashes
        from travel.rasp.tasks.suburban_events import update_expected_suburban_events
        from travel.rasp.tasks.suburban_events import update_expected_suburban_events_after_prepare_all
        from travel.rasp.tasks.suburban_events import update_suburban_events
        from travel.rasp.tasks.suburban_events import update_movista_raw_cancels
        from travel.rasp.tasks.suburban_events import update_movista_matched_cancels

        scripts = {
            # Мониторинг получения данных от РЖД
            'monitoring': suburban_events.run,

            # Пересчет событий/прогноза Электричек
            'update_suburban_events': update_suburban_events.lock_aware_run,
            'update_expected_suburban_events': update_expected_suburban_events.lock_aware_run,
            'update_expected_suburban_events_after_prepare_all': update_expected_suburban_events_after_prepare_all.lock_aware_run,
            'clean_suburban_events': clean_suburban_events.lock_aware_run,
            'update_company_crashes': update_company_crashes.lock_aware_run,

            'update_movista_raw_cancels': update_movista_raw_cancels.lock_aware_run,
            'update_movista_matched_cancels': update_movista_matched_cancels.lock_aware_run,
        }

        with self.report_progress(source=self.Parameters.source, environment=self.Parameters.env_type):
            scripts[self.Parameters.source]()
