# encoding: utf-8

import logging
import os
import shutil
import tempfile

from sandbox import sdk2

from sandbox.common.types.resource import State
from sandbox.projects.common import binary_task
from sandbox.projects.common.ya_deploy import release_integration
from sandbox.projects.rasp.resource_types import RaspSuggestsObjsDataResource, RaspSuggestsIdConverterResource, MessagePackResource
from sandbox.projects.rasp.utils.email_notifications import EmailNotificationMixin, use_email_notification_params, RASP_GROUP
from sandbox.projects.rasp.utils.geobase import GeobaseMixin, use_geobase_params
from sandbox.projects.rasp.utils.juggler import JugglerNotificationMixin, use_juggler_notification_params
from sandbox.projects.rasp.utils.solomon import SolomonProgressMixin
from sandbox.projects.rasp.utils.rasp_scripts import RaspScriptsTask, use_rasp_scripts_params

log = logging.getLogger(__name__)
log.setLevel(logging.INFO)

SUGGESTS_RES_TTL = 7


class RaspSuggestsGenerate(
    release_integration.ReleaseToYaDeployTask2,
    binary_task.LastBinaryTaskRelease,
    sdk2.Task,
    EmailNotificationMixin,
    JugglerNotificationMixin,
    RaspScriptsTask,
    SolomonProgressMixin,
    GeobaseMixin
):
    class Requirements(sdk2.Task.Requirements):
        class Caches(sdk2.Requirements.Caches):  # enable multislot https://wiki.yandex-team.ru/sandbox/clients/#client-tags-multislot
            pass  # means that task do not use any shared caches

        cores = 4  # cores number is required to enable multislot
        ram = 16 * 1024

    class Parameters(sdk2.Task.Parameters):
        kill_timeout = 3600 * 3

        ext_params = binary_task.binary_release_parameters(stable=True)  # нужно для работы LastBinaryTaskRelease

        with sdk2.parameters.Group('General') as mds_block:
            search_stat = sdk2.parameters.Resource(
                'Search stat', required=False,
                description='Using last search stat MESSAGE_PACK_RESOURCE if not specified'
            )
            notifications_group = sdk2.parameters.String('Notification group', required=True, default=RASP_GROUP)
            ignore_hidden_field = sdk2.parameters.Bool('Ignore hidden field', required=False, default=False)
            service_name = sdk2.parameters.String('Service name', required=True, default='rasp')

        _rasp_scripts_params = use_rasp_scripts_params()
        _email_notification_params = use_email_notification_params()
        _juggler_notification_params = use_juggler_notification_params()
        _geobase_params = use_geobase_params()

    def on_save(self):
        binary_task.LastBinaryTaskRelease.on_save(self)  # нужно для работы LastBinaryTaskRelease
        self.add_email_notifications(notifications_group=self.Parameters.notifications_group)
        self.add_juggler_notifications(environment=self.Parameters.env_type)

    def set_task_environ(self):
        os.environ['DJANGO_SETTINGS_MODULE'] = 'travel.rasp.suggests_tasks.bin.sandbox.local_settings'
        os.environ['RASP_GEOBASE_DATA_PATH'] = self.geobase_path

        if self.Parameters.ignore_hidden_field:
            os.environ['RASP_IGNORE_HIDDEN_FIELD'] = '1'

    def on_enqueue(self):
        if not self.Parameters.search_stat:
            self.Parameters.search_stat = MessagePackResource.find(
                state=State.READY,
                owner=self.Parameters.owner,
                attrs=dict(
                    env_type=self.Parameters.env_type,
                    suggests_file='stat.msgpack'
                )).first()

    def on_release(self, additional_parameters):
        super(RaspSuggestsGenerate, self).on_release(additional_parameters)
        self.mark_released_resources(additional_parameters['release_status'], ttl=SUGGESTS_RES_TTL)

    def on_execute(self):
        self.download_geobase()
        self.set_common_environ()

        from travel.rasp.suggests_tasks.suggests.generate.db import MaintenanceInProcess, raise_if_maintenance_in_process
        from travel.rasp.suggests_tasks.suggests.generate.generate import run
        from travel.rasp.suggests_tasks.suggests.storage import Storage
        from travel.rasp.suggests_tasks.suggests.msgpack_utils import (
            reformat_msgpack, reformat_idconverter, reformat_objs_data
        )

        log.info("Initializing storage & stat_file")
        if not self.Parameters.search_stat:
            raise sdk2.task.common_errors.ResourceNotFound('Search stat resource is not found')

        storage = Storage(tempfile.gettempdir())
        data = sdk2.ResourceData(self.Parameters.search_stat)
        shutil.copy(str(data.path), os.path.join(storage.data_path, storage.stat_file))

        log.info("Running suggests data generation")
        try:
            with self.report_progress(source='generate_suggests', environment=self.Parameters.env_type):
                run(storage.data_path, skip_mds=True)
            raise_if_maintenance_in_process()
        except MaintenanceInProcess:
            log.info('Maintenance is in process. Quitting')
            return

        log.info("Saving suggests data as resources")
        idconverter = RaspSuggestsIdConverterResource(self, 'ID Converter', storage.id_conv_file,
                                                      env_type=self.Parameters.env_type,
                                                      service_name=self.Parameters.service_name,
                                                      suggests_file=storage.id_conv_file)

        objs_data = RaspSuggestsObjsDataResource(self, 'Objects Data', storage.objs_data_file,
                                                 env_type=self.Parameters.env_type,
                                                 service_name=self.Parameters.service_name,
                                                 suggests_file=storage.objs_data_file)

        with open(os.path.join(storage.data_path, storage.id_conv_file), 'r') as f:
            data = sdk2.ResourceData(idconverter)
            data.path.write_bytes(reformat_msgpack(f.read(), reformat_idconverter))
            data.ready()

        with open(os.path.join(storage.data_path, storage.objs_data_file), 'r') as f:
            data = sdk2.ResourceData(objs_data)
            data.path.write_bytes(reformat_msgpack(f.read(), reformat_objs_data))
            data.ready()
